package com.atlassian.jira.bc.user.search;

import com.atlassian.crowd.embedded.api.User;
import com.atlassian.jira.user.UserFilter;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Set;

/**
 * Optional parameters to restrict or control a user search.
 * <p>
 * This allows you to include or exclude active and inactive users,
 * allow or disallow empty search queries and avoid returning stale data.
 * <p>
 * Implementations which use this class may return "eventually consistent"/stale/cached
 * data to improve performance.<br>
 * If that's not what you expect as for example you don't want to return stale data in:<br>
 * - security/permission context to avoid security problems,<br>
 * - notification context to avoid sending-off notification to users who should not get information about something<br>
 * you can set {@link UserSearchParams#forceStrongConsistency forceStrongConsistency}
 * to true to force the conforming implementation to return "strongly consistent"/fresh data.
 *
 * @since v5.1.5
 */
public class UserSearchParams {

    private static final Logger log = LoggerFactory.getLogger(UserSearchParams.class);

    public static final int MAXIMUM_RESULTS = 100;

    @Deprecated
    public static final UserSearchParams ACTIVE_USERS_IGNORE_EMPTY_QUERY = new UserSearchParams(false, true, false,
            false, null, null);
    @Deprecated
    public static final UserSearchParams ACTIVE_USERS_ALLOW_EMPTY_QUERY = new UserSearchParams(true, true, false,
            false, null, null);

    public static final UserSearchParams LIMITED_ACTIVE_USERS_IGNORE_EMPTY_QUERY = new UserSearchParams(false, true, false,
            false, null, null, MAXIMUM_RESULTS, true, false);
    public static final UserSearchParams LIMITED_ACTIVE_USERS_ALLOW_EMPTY_QUERY = new UserSearchParams(true, true, false,
            false, null, null, MAXIMUM_RESULTS, true, false);


    private final boolean allowEmptyQuery;

    private final boolean includeActive;

    private final boolean includeInactive;

    /**
     * Indicate whether the search would apply the query to email address as well.
     *
     * @since v6.2
     */
    private final boolean canMatchEmail;

    /**
     * The additional filters to be applied to the search.
     *
     * @since v6.2
     */

    private final UserFilter userFilter;
    /**
     * The list of project ids to be used in conjunction with the roles in {@link #userFilter}.
     *
     * @since v6.2
     */

    private final Set<Long> projectIds;

    /**
     * Filter that runs after everything else.
     *
     * @since v7.0
     * @deprecated Using postProcessingFilter field may cause performance issues. It will be removed in Jira version 9.
     */
    @Nonnull
    @Deprecated
    private final Predicate<User> postProcessingFilter;

    /**
     * The maximum number of results to retrieve.
     *
     * @since v7.0
     */
    private final Integer maxResults;

    /**
     * Whether the results must be sorted.  By default sorting is active, but it can be disabled for a performance boost.
     *
     * @since v7.0
     */
    private final boolean sorted;

    /**
     * A flag that indicates whether a browser users permission check will be performed.  If false,
     * the logged in user must have browser users permission to list users or else an empty list is always returned.
     * If true, searches are not restricted by the logged in user's permissions.
     *
     * @since v7.0
     */
    private final boolean ignorePermissionCheck;

    /**
     * Indicates whether the search must avoid returning stale data.
     *
     * @since v9.0
     */
    private final boolean forceStrongConsistency;

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery if true, empty queries that would retrieve all results are allowed.
     * @param includeActive   active users are searched.
     * @param includeInactive inactive users are searched.
     * @deprecated Using constructor without maxResults param is discouraged.
     */
    @Deprecated
    public UserSearchParams(boolean allowEmptyQuery, boolean includeActive, boolean includeInactive) {
        this(allowEmptyQuery, includeActive, includeInactive, false, null, null);
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery if true, empty queries that would retrieve all results are allowed.
     * @param includeActive   active users are searched.
     * @param includeInactive inactive users are searched.
     * @param canMatchEmail   if search applies to email address as well.
     * @param userFilter      filter users by groups or roles.
     * @param projectIds      the list of project ids to be used in conjunction with the roles in user filter.
     * @since v6.2
     * @deprecated Using constructor without limit parameter is discouraged.
     */
    @Deprecated
    public UserSearchParams(boolean allowEmptyQuery, boolean includeActive, boolean includeInactive,
                            boolean canMatchEmail,
                            final UserFilter userFilter, final Set<Long> projectIds) {
        this(allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter,
                projectIds, Predicates.<User>alwaysTrue());
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery      if true, empty queries that would retrieve all results are allowed.
     * @param includeActive        active users are searched.
     * @param includeInactive      inactive users are searched.
     * @param canMatchEmail        if search applies to email address as well.
     * @param userFilter           filter users by groups or roles.
     * @param projectIds           the list of project ids to be used in conjunction with the roles in user filter.
     * @param postProcessingFilter a filter applied after search results are retrieved.  Use this for more complex
     *                             filter logic that can't be expressed in <code>userFilter</code> and the other parameters.
     * @since v7.0
     * @deprecated Using constructor without maxResults param is discouraged. Using postProcessingFilter parameter may cause performance issues.
     *         postProcessingFilter parameter will be removed in Jira version 9.
     */
    @Deprecated
    public UserSearchParams(boolean allowEmptyQuery, boolean includeActive, boolean includeInactive,
                            boolean canMatchEmail,
                            final UserFilter userFilter, final Set<Long> projectIds,
                            final Predicate<User> postProcessingFilter) {
        this(allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter,
                projectIds, postProcessingFilter, null, true, false);
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery      if true, empty queries that would retrieve all results are allowed.
     * @param includeActive        active users are searched.
     * @param includeInactive      inactive users are searched.
     * @param canMatchEmail        if search applies to email address as well.
     * @param userFilter           filter users by groups or roles.
     * @param projectIds           the list of project ids to be used in conjunction with the roles in user filter.
     * @param postProcessingFilter a filter applied after search results are retrieved.  Use this for more complex
     *                             filter logic that can't be expressed in <code>userFilter</code> and the other parameters.
     * @param maxResults           the maximum number of results to retrieve.  Use <code>null</code> for unlimited results.
     * @since v7.0
     * @deprecated Using postProcessingFilter parameter may cause performance issues. It will be removed in Jira version 9.
     *         Passing null as maxResults param is discouraged. maxResults parameter will be removed as of Jira version 9.
     *         Please use constructor with limit parameter instead.
     */
    @Deprecated
    public UserSearchParams(boolean allowEmptyQuery, boolean includeActive, boolean includeInactive,
                            boolean canMatchEmail,
                            final UserFilter userFilter, final Set<Long> projectIds,
                            final Predicate<User> postProcessingFilter,
                            Integer maxResults) {
        this(allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter, projectIds,
                postProcessingFilter, maxResults, true, false);
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery      if true, empty queries that would retrieve all results are allowed.
     * @param includeActive        active users are searched.
     * @param includeInactive      inactive users are searched.
     * @param canMatchEmail        if search applies to email address as well.
     * @param userFilter           filter users by groups or roles.
     * @param projectIds           the list of project ids to be used in conjunction with the roles in user filter.
     * @param postProcessingFilter a filter applied after search results are retrieved.  Use this for more complex
     *                             filter logic that can't be expressed in <code>userFilter</code> and the other parameters.
     * @param maxResults           the maximum number of results to retrieve.  Use <code>null</code> for unlimited results.
     * @param sorted               if true, results are guaranteed to be sorted.  If false, results may be unsorted which might
     *                             have better performance.
     * @since v7.0
     * @deprecated Using postProcessingFilter parameter may cause performance issues. It will be removed in Jira version 9.
     *         Passing null as maxResults param is discouraged. maxResults parameter will be removed as of Jira version 9.
     *         Please use constructor with limit parameter instead.
     */
    @Deprecated
    public UserSearchParams(boolean allowEmptyQuery, boolean includeActive, boolean includeInactive,
                            boolean canMatchEmail,
                            final UserFilter userFilter, final Set<Long> projectIds,
                            final Predicate<User> postProcessingFilter,
                            Integer maxResults,
                            boolean sorted) {
        this(allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter, projectIds,
                postProcessingFilter, maxResults, sorted, false);
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery       if true, empty queries that would retrieve all results are allowed.
     * @param includeActive         active users are searched.
     * @param includeInactive       inactive users are searched.
     * @param canMatchEmail         if search applies to email address as well.
     * @param userFilter            filter users by groups or roles.
     * @param projectIds            the list of project ids to be used in conjunction with the roles in user filter.
     * @param postProcessingFilter  a filter applied after search results are retrieved.  Use this for more complex
     *                              filter logic that can't be expressed in <code>userFilter</code> and the other parameters.
     * @param maxResults            the maximum number of results to retrieve.  Use <code>null</code> for unlimited results.
     * @param sorted                if true, results are guaranteed to be sorted.  If false, results may be unsorted which might
     *                              have better performance.
     * @param ignorePermissionCheck a flag that indicates whether a browser users permission check will be performed.
     * @since v7.0
     * @deprecated Using postProcessingFilter parameter may cause performance issues. It will be removed in Jira version 9.
     *         Passing null as maxResults param is discouraged. maxResults parameter will be removed as of Jira version 9.
     *         Please use constructor with limit parameter instead.
     */
    @Deprecated
    public UserSearchParams(boolean allowEmptyQuery, boolean includeActive, boolean includeInactive,
                            boolean canMatchEmail,
                            final UserFilter userFilter, final Set<Long> projectIds,
                            final Predicate<User> postProcessingFilter,
                            Integer maxResults,
                            boolean sorted,
                            boolean ignorePermissionCheck) {
        this(allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter, projectIds,
                postProcessingFilter, maxResults, sorted, ignorePermissionCheck, false);
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery       if true, empty queries that would retrieve all results are allowed.
     * @param includeActive         active users are searched.
     * @param includeInactive       inactive users are searched.
     * @param canMatchEmail         if search applies to email address as well.
     * @param userFilter            filter users by groups or roles.
     * @param projectIds            the list of project ids to be used in conjunction with the roles in user filter.
     * @param limit                 the maximum number of results to retrieve. The maximum allowed is 100. If higher number is provided it will be set to 100.
     * @param sorted                if true, results are guaranteed to be sorted.  If false, results may be unsorted which might
     *                              have better performance.
     * @param ignorePermissionCheck a flag that indicates whether a browser users permission check will be performed.
     * @since v8.20
     */
    public UserSearchParams(boolean allowEmptyQuery,
                            boolean includeActive,
                            boolean includeInactive,
                            boolean canMatchEmail,
                            final UserFilter userFilter,
                            final Set<Long> projectIds,
                            int limit,
                            boolean sorted,
                            boolean ignorePermissionCheck) {
        this(allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter, projectIds, limit, sorted, ignorePermissionCheck, false);
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery        if true, empty queries that would retrieve all results are allowed.
     * @param includeActive          active users are searched.
     * @param includeInactive        inactive users are searched.
     * @param canMatchEmail          if search applies to email address as well.
     * @param userFilter             filter users by groups or roles.
     * @param projectIds             the list of project ids to be used in conjunction with the roles in user filter.
     * @param limit                  the maximum number of results to retrieve. The maximum allowed is 100. If higher number is provided it will be set to 100.
     * @param sorted                 if true, results are guaranteed to be sorted.  If false, results may be unsorted which might
     *                               have better performance.
     * @param ignorePermissionCheck  a flag that indicates whether a browser users permission check will be performed.
     * @param forceStrongConsistency whether the search must avoid returning stale data.
     *                               Use only if really necessary as it may cripple performance.
     * @since v9.0
     */
    public UserSearchParams(
            final boolean allowEmptyQuery,
            final boolean includeActive,
            final boolean includeInactive,
            final boolean canMatchEmail,
            final UserFilter userFilter,
            final Set<Long> projectIds,
            int limit,
            final boolean sorted,
            final boolean ignorePermissionCheck,
            final boolean forceStrongConsistency) {
        this.allowEmptyQuery = allowEmptyQuery;
        this.includeActive = includeActive;
        this.includeInactive = includeInactive;
        this.canMatchEmail = canMatchEmail;
        this.userFilter = userFilter;
        this.projectIds = projectIds;
        this.postProcessingFilter = Predicates.alwaysTrue();
        if (limit > MAXIMUM_RESULTS) {
            log.trace("limit {} exceeded the maximum. Setting limit to max: {}", limit, MAXIMUM_RESULTS);
            this.maxResults = MAXIMUM_RESULTS;
        } else {
            this.maxResults = limit;
        }
        this.sorted = sorted;
        this.ignorePermissionCheck = ignorePermissionCheck;
        this.forceStrongConsistency = forceStrongConsistency;
    }

    /**
     * Creates user search params.
     *
     * @param allowEmptyQuery        if true, empty queries that would retrieve all results are allowed.
     * @param includeActive          active users are searched.
     * @param includeInactive        inactive users are searched.
     * @param canMatchEmail          if search applies to email address as well.
     * @param userFilter             filter users by groups or roles.
     * @param projectIds             the list of project ids to be used in conjunction with the roles in user filter.
     * @param postProcessingFilter   a filter applied after search results are retrieved.  Use this for more complex
     *                               filter logic that can't be expressed in <code>userFilter</code> and the other parameters.
     * @param maxResults             the maximum number of results to retrieve.  Use <code>null</code> for unlimited results.
     * @param sorted                 if true, results are guaranteed to be sorted.  If false, results may be unsorted which might
     *                               have better performance.
     * @param ignorePermissionCheck  a flag that indicates whether a browser users permission check will be performed.
     * @param forceStrongConsistency whether the search must avoid returning stale data.
     *                               Use only if really necessary as it may cripple performance.
     * @since v9.0
     * @deprecated Remove after {@link UserSearchParams#postProcessingFilter} is removed in Jira 9.
     */
    @Deprecated
    private UserSearchParams(
            final boolean allowEmptyQuery,
            final boolean includeActive,
            final boolean includeInactive,
            final boolean canMatchEmail,
            final UserFilter userFilter,
            final Set<Long> projectIds,
            final Predicate<User> postProcessingFilter,
            Integer maxResults,
            final boolean sorted,
            final boolean ignorePermissionCheck,
            final boolean forceStrongConsistency) {
        this.allowEmptyQuery = allowEmptyQuery;
        this.includeActive = includeActive;
        this.includeInactive = includeInactive;
        this.canMatchEmail = canMatchEmail;
        this.userFilter = userFilter;
        this.projectIds = projectIds;
        if (postProcessingFilter == null)
            this.postProcessingFilter = Predicates.alwaysTrue();
        else
            this.postProcessingFilter = postProcessingFilter;

        this.maxResults = maxResults;
        this.sorted = sorted;
        this.ignorePermissionCheck = ignorePermissionCheck;
        this.forceStrongConsistency = forceStrongConsistency;
    }

    public boolean allowEmptyQuery() {
        return allowEmptyQuery;
    }

    public boolean includeActive() {
        return includeActive;
    }

    public boolean includeInactive() {
        return includeInactive;
    }

    public boolean canMatchEmail() {
        return canMatchEmail;
    }

    public UserFilter getUserFilter() {
        return userFilter;
    }

    public Set<Long> getProjectIds() {
        return projectIds;
    }

    /**
     * @since v7.0
     */
    public boolean ignorePermissionCheck() {
        return ignorePermissionCheck;
    }

    /**
     * @since v7.0
     * @deprecated Using postProcessingFilter field may cause performance issues. It will be removed in Jira version 9.
     */
    @Nonnull
    @Deprecated
    public Predicate<User> getPostProcessingFilter() {
        return postProcessingFilter;
    }

    /**
     * @since v7.0
     */
    public Integer getMaxResults() {
        return maxResults;
    }

    /**
     * @since v7.0
     */
    public boolean isSorted() {
        return sorted;
    }

    /**
     * @since v9.0
     */
    public boolean isForcingStrongConsistency() {
        return forceStrongConsistency;
    }

    @Override
    public boolean equals(final Object o) {
        return EqualsBuilder.reflectionEquals(this, o);
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Deprecated
    public static Builder builder() {
        return new Builder();
    }

    /**
     * @param limit the maximum number of results to retrieve using constructed UserSearchParams.
     *              The maximum allowed is 100. If higher number is provided it will be set to 100 in resulting UserSearchParams.
     * @since v8.20
     */
    public static Builder builder(int limit) {
        return new Builder(limit);
    }

    public static Builder builder(UserSearchParams prototype) {
        return new Builder(prototype);
    }

    public static class Builder {
        private boolean allowEmptyQuery = false;
        private boolean includeActive = true;
        private boolean includeInactive = false;
        private boolean canMatchEmail = false;
        private UserFilter userFilter = null;
        private Set<Long> projectIds = null;
        @Deprecated
        private Predicate<User> postProcessingFilter;
        private Integer maxResults;
        private boolean sorted = true;
        private boolean ignorePermissionCheck = false;
        private boolean forceStrongConsistency = false;

        @Deprecated
        public Builder() {
        }

        /**
         * @param limit the maximum number of results to retrieve using constructed UserSearchParams.
         *              The maximum allowed is 100. If higher number is provided it will be set to 100 in resulting UserSearchParams.
         * @since v8.20
         */
        public Builder(int limit) {
            this.maxResults = limit;
        }


        private Builder(UserSearchParams prototype) {
            this.allowEmptyQuery = prototype.allowEmptyQuery;
            this.includeActive = prototype.includeActive;
            this.includeInactive = prototype.includeInactive;
            this.canMatchEmail = prototype.canMatchEmail;
            this.userFilter = prototype.userFilter;
            this.projectIds = prototype.projectIds;
            this.postProcessingFilter = prototype.postProcessingFilter;
            this.maxResults = prototype.maxResults;
            this.sorted = prototype.sorted;
            this.ignorePermissionCheck = prototype.ignorePermissionCheck;
            this.forceStrongConsistency = prototype.forceStrongConsistency;
        }

        public UserSearchParams build() {
            return new UserSearchParams(
                    allowEmptyQuery, includeActive, includeInactive, canMatchEmail, userFilter, projectIds,
                    postProcessingFilter, maxResults, sorted, ignorePermissionCheck, forceStrongConsistency);
        }

        /**
         * @param allowEmptyQuery if true, empty queries that would retrieve all results are allowed.
         */
        public Builder allowEmptyQuery(boolean allowEmptyQuery) {
            this.allowEmptyQuery = allowEmptyQuery;
            return this;
        }

        /**
         * @param includeActive active users are searched.
         */
        public Builder includeActive(boolean includeActive) {
            this.includeActive = includeActive;
            return this;
        }

        /**
         * @param includeInactive inactive users are searched.
         */
        public Builder includeInactive(boolean includeInactive) {
            this.includeInactive = includeInactive;
            return this;
        }

        /**
         * @param canMatchEmail if search applies to email address as well.
         */
        public Builder canMatchEmail(boolean canMatchEmail) {
            this.canMatchEmail = canMatchEmail;
            return this;
        }

        /**
         * @param userFilter filter users by groups or roles.
         * @since v6.2
         */
        public Builder filter(UserFilter userFilter) {
            this.userFilter = userFilter;
            return this;
        }

        /**
         * @param projectIds the list of project ids to be used in conjunction with the roles in user filter.
         * @since v6.2
         */
        public Builder filterByProjectIds(Collection<Long> projectIds) {
            this.projectIds = projectIds == null ? null : ImmutableSet.copyOf(projectIds);
            return this;
        }

        /**
         * @param postProcessingFilter a filter applied after search results are retrieved.  Use this for more complex
         *                             filter logic that can't be expressed in a {@link UserFilter} or other parameters.
         * @since v7.0
         * @deprecated Using postProcessingFilter parameter may cause performance issues. It will be removed in Jira version 9.
         */
        @Deprecated
        public Builder filter(Predicate<User> postProcessingFilter) {
            this.postProcessingFilter = postProcessingFilter;
            return this;
        }

        /**
         * @param maxResults the maximum number of results to retrieve.
         * @since v7.0
         * @deprecated Use {@link #limitResults(int) } instead.
         */
        @Deprecated
        public Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        /**
         * @param limit the maximum number of results to retrieve. The maximum allowed is 100.
         *              If higher number is provided it will be set to 100 in resulting UserSearchParams.
         * @since v8.20
         */
        public Builder limitResults(int limit) {
            this.maxResults = limit;
            return this;
        }

        /**
         * @param sorted true to ensure results are sorted, false if results may be unsorted.  Turning sorting off
         *               may have better performance.
         * @since v7.0
         */
        public Builder sorted(boolean sorted) {
            this.sorted = sorted;
            return this;
        }

        /**
         * @param ignorePermissionCheck a flag indicating whether the permission checking has already been
         *                              performed.  If false (the default), the logged in user must have browser users permission to list
         *                              users or else an empty list is always returned.
         *                              If true, searches are not restricted by the logged in user's permissions.
         * @since v7.0
         */
        public Builder ignorePermissionCheck(boolean ignorePermissionCheck) {
            this.ignorePermissionCheck = ignorePermissionCheck;
            return this;
        }

        /**
         * @param forceStrongConsistency whether the search must avoid returning stale data.
         *                               Use only if really necessary as it may cripple performance.
         * @since v9.0
         */
        public Builder forceStrongConsistency(boolean forceStrongConsistency) {
            this.forceStrongConsistency = forceStrongConsistency;

            return this;
        }
    }
}
