package com.atlassian.jira.bc.safeguards.issue.comment;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.exception.CommentsPerIssueLimitExceededException;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

/**
 * Validates if the comments per issue limit is not reached.
 *
 * @since 9.0.0
 */
@ExperimentalApi
public interface CommentsPerIssueLimitValidator {

    /**
     * Checks if limit is not reached yet
     * (eg. to determine if new entity can be created).
     *
     * If validation fails, an error with description is added to error collection.
     *
     * @param issue entity analysed for comment number
     * @param author comment creator
     * @param commentBody comment body
     * @return true if limit is not reached
     */
    boolean isValid(@Nonnull Issue issue, @Nullable ApplicationUser author, @Nonnull String commentBody, @Nonnull ErrorCollection errorCollection);

    /**
     * Validates comment limits for all provided issues and returns issues that failed.
     * This method is a recommended way to validate a collection of issues due to performance optimizations.
     * This method doesn't trigger any notification or logging.
     *
     * @param issues list of issues
     * @return list of issues for which validation failed
     */
    @Nonnull
    List<Issue> getInvalidIssues(@Nonnull List<Issue> issues);

    /**
     * Checks if limit is not reached yet
     * (eg. to determine if new entity can be created).
     *
     * @param issue entity analysed for comment number
     * @param author comment creator
     * @param commentBody comment body
     * @throws CommentsPerIssueLimitExceededException if validation fails
     */
    void validate(@Nonnull Issue issue, @Nullable ApplicationUser author, @Nonnull String commentBody) throws CommentsPerIssueLimitExceededException;

    /**
     * Returns the value of the limit as configured at the time of calling,
     * @return the configured limit
     */
    long getLimit();
}
