package com.atlassian.jira.bc.project.component;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.user.ApplicationUser;
import com.opensymphony.util.TextUtils;
import org.ofbiz.core.entity.GenericValue;

import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;

@PublicApi
public class MutableProjectComponent implements ProjectComponent {

    private Long id;
    private String name;
    private String lead;
    private String description;
    private Long projectId;
    private long assigneeType;
    private boolean archived;
    private boolean deleted;

    /**
     * Temporary, during migration to use ProjectComponent object.
     */
    private GenericValue genericValue;

    /**
     * Please, use {@link #MutableProjectComponent(Long, String, String, String, long, Long, boolean)}
     */
    @Deprecated
    public MutableProjectComponent(Long id, String name, String description, String lead, long assigneeType, Long projectId) {
        this(id, name, description, lead, assigneeType, projectId, false);
    }

    /**
     * Please, use {@link #MutableProjectComponent(Long, String, String, String, long, Long, boolean, boolean)}
     */
    @Deprecated
    public MutableProjectComponent(Long id, String name, String description, String lead, long assigneeType, Long projectId, boolean archived) {
        this(id, name, description, lead, assigneeType, projectId, archived, false);
    }

    public MutableProjectComponent(Long id, String name, String description, String lead, long assigneeType, Long projectId, boolean archived, boolean deleted) {
        this.id = id;
        setDescription(description);
        this.lead = lead;
        this.name = name;
        this.projectId = projectId;
        this.assigneeType = assigneeType;
        this.archived = archived;
        this.deleted = deleted;
    }

    @Override
    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    @Override
    public String getDescription() {
        return description;
    }

    /**
     * Set description to give value. If the value is an empty string, it will be set to null
     *
     * @param description description to set it to
     */
    public void setDescription(final String description) {
        this.description = TextUtils.stringSet(description) ? description : null;
    }

    @Override
    public String getLead() {
        return lead;
    }

    @Override
    public ApplicationUser getComponentLead() {
        return ComponentAccessor.getUserManager().getUserByKey(getLead());
    }

    public void setLead(final String lead) {
        this.lead = lead;
    }

    @Override
    public String getName() {
        return name;
    }

    public void setName(final String name) {
        this.name = name;
    }

    @Override
    public Long getProjectId() {
        return projectId;
    }

    @Override
    public long getAssigneeType() {
        return this.assigneeType;
    }

    public void setAssigneeType(final long assigneeType) {
        this.assigneeType = assigneeType;
    }

    @Override
    public GenericValue getGenericValue() {
        return genericValue;
    }

    void setGenericValue(final GenericValue genericValue) {
        this.genericValue = genericValue;
    }

    @Override
    public boolean isArchived() {
        return archived;
    }

    public void setArchived(final boolean archived) {
        this.archived = archived;
    }

    @Override
    public boolean isDeleted() {
        return deleted;
    }

    public void setDeleted(final boolean deleted) {
        this.deleted = deleted;
    }

    /**
     * Returns true if same object, instance of MutableProjectComponent and equal ID.
     *
     * @param o object to compare with
     * @return true if same object, instance of MutableProjectComponent and equal ID; false otherwise
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof MutableProjectComponent)) {
            return false;
        }
        Long thatId = ((MutableProjectComponent) o).getId();
        return thatId != null && thatId.equals(id);
    }

    /**
     * Returns the hash code of the ID
     *
     * @return hash code of the ID
     */
    public int hashCode() {
        return id.hashCode();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("MutableProjectComponent { name='");
        sb.append(name);
        sb.append("', description='");
        sb.append(description == null ? "" : description);
        sb.append("'");
        sb.append(", lead='");
        sb.append(lead == null ? "" : lead);
        sb.append(", assigneeType='").append(assigneeType);
        sb.append(", archived='").append(archived);
        sb.append("' }");
        return sb.toString();
    }

    public boolean equalsName(MutableProjectComponent component) {
        return getName().equalsIgnoreCase(component.getName());
    }

    /**
     * Copy given component
     *
     * @param c component to copy
     * @return new instance of MutableProjectComponent set with values of the given component
     */
    public static MutableProjectComponent copy(final ProjectComponent c) {
        MutableProjectComponent mutableProjectComponent = new MutableProjectComponent(c.getId(), c.getName(), c.getDescription(), c.getLead(), c.getAssigneeType(), c.getProjectId(), c.isArchived(), c.isDeleted());
        mutableProjectComponent.setGenericValue(c.getGenericValue());
        return mutableProjectComponent;
    }

    /**
     * Copy given collection of components
     *
     * @param c collection of components to copy
     * @return new instance collection of MutableProjectComponent objects set with values of the given components
     */
    public static Collection<MutableProjectComponent> copy(final Collection<ProjectComponent> c) {
        if (c == null) {
            return Collections.emptyList();
        }
        return c.stream()
                .map(MutableProjectComponent::copy)
                .collect(Collectors.toList());

    }

}
