package com.atlassian.jira.bc.project;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.security.plugin.ProjectPermissionKey;

import java.util.EnumMap;
import java.util.Map;

import static com.atlassian.jira.bc.project.ProjectAction.PredicateContext;
import static com.atlassian.jira.bc.project.ProjectAction.VIEW_ARCHIVED_PROJECT;
import static com.atlassian.jira.bc.project.ProjectAction.VIEW_PROJECT;
import static com.atlassian.jira.config.properties.APKeys.PROJECT_PERMISSION_HELPER_BETA_FIX_ENABLED;
import static com.atlassian.jira.security.Permissions.ADMINISTER;
import static com.atlassian.jira.security.Permissions.BROWSE;
import static com.atlassian.jira.security.Permissions.PROJECT_ADMIN;
import static com.atlassian.jira.security.Permissions.isGlobalPermission;
import static com.atlassian.jira.security.ProjectWidePermission.ALL_ISSUES;

class ProjectActionPermissionHelper {

    private static ProjectActionPermissionHelper INSTANCE = new ProjectActionPermissionHelper();

    private static final int[] ADMIN_PERMISSIONS = new int[]{ADMINISTER, PROJECT_ADMIN};
    private static final int[] BROWSE_PERMISSION = new int[]{BROWSE};
    private static final int[] EMPTY_PERM = new int[]{};

    private static final Map<ProjectAction, int[]> PARTIAL_PERMISSIONS;
    private static final Map<ProjectAction, int[]> PROJECT_WIDE_PERMISSIONS;

    private static Boolean isFeatureEnabled = null;

    static {
        PARTIAL_PERMISSIONS = new EnumMap<>(ProjectAction.class);
        PARTIAL_PERMISSIONS.put(VIEW_PROJECT, ADMIN_PERMISSIONS);
        PARTIAL_PERMISSIONS.put(VIEW_ARCHIVED_PROJECT, ADMIN_PERMISSIONS);

        PROJECT_WIDE_PERMISSIONS = new EnumMap<>(ProjectAction.class);
        PROJECT_WIDE_PERMISSIONS.put(VIEW_PROJECT, BROWSE_PERMISSION);
        PROJECT_WIDE_PERMISSIONS.put(VIEW_ARCHIVED_PROJECT, BROWSE_PERMISSION);
    }

    private ProjectActionPermissionHelper() {

    }

    public static ProjectActionPermissionHelper getInstance() {
        return INSTANCE;
    }

    public int[] getPartialPermissions(ProjectAction action) {
        return isFixEnabled() ? PARTIAL_PERMISSIONS.getOrDefault(action, action.getPermissions())
                : action.getPermissions();
    }

    public int[] getProjectWidePermissions(ProjectAction action) {
        return isFixEnabled() ? PROJECT_WIDE_PERMISSIONS.getOrDefault(action, EMPTY_PERM)
                : EMPTY_PERM;
    }


    public boolean checkPermissions(ProjectAction action, PredicateContext predicateContext) {
        for (int permission : getPartialPermissions(action)) {
            if (checkPartialPermissions(predicateContext, permission)) {
                return true;
            }
        }
        for (int permission : getProjectWidePermissions(action)) {
            if (checkProjectWidePermissions(predicateContext, permission)) {
                return true;
            }
        }
        return false;
    }

    private boolean checkPartialPermissions(PredicateContext predicateContext, int permission) {
        if (isGlobalPermission(permission)) {
            return predicateContext.getPermissionManager().hasPermission(permission, predicateContext.getUser());
        } else {
            return predicateContext.getPermissionManager().hasPermission(new ProjectPermissionKey(permission),
                    predicateContext.getProject(), predicateContext.getUser());
        }
    }

    private boolean checkProjectWidePermissions(PredicateContext predicateContext, int permission) {
        return predicateContext
                .getPermissionManager()
                .hasProjectWidePermission(
                        new ProjectPermissionKey(permission),
                        predicateContext.getProject(),
                        predicateContext.getUser())
                .equals(ALL_ISSUES);
    }

    boolean isFixEnabled() {
        if(isFeatureEnabled == null) {
            ApplicationProperties applicationProperties = ComponentAccessor.getComponentSafely(ApplicationProperties.class).orElse(null);
            if(applicationProperties != null) {
                isFeatureEnabled = applicationProperties.getOption(PROJECT_PERMISSION_HELPER_BETA_FIX_ENABLED);
            } else {
                return false;
            }
        }
        return isFeatureEnabled;
    }
}
