package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalSpi;
import com.atlassian.jira.task.context.Context;
import com.atlassian.jira.task.context.Contexts;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Contains original and target values of the user property being changed.
 * Also contains {@code context} to report progress to.
 */
@ExperimentalSpi
public class UserPropertyChangeParameter implements TaskContextAware {
    private final String original;
    private final String target;
    private final Context context;

    public UserPropertyChangeParameter(@Nonnull String original, @Nonnull String target, @Nonnull Context context) {
        this.original = requireNonNull(original);
        this.target = requireNonNull(target);
        this.context = context;
    }

    /**
     * Original value of the parameter, e.g. user key/username, to be changed.
     */
    @Nonnull
    public String getOriginal() {
        return original;
    }

    /**
     * Target parameter value, e.g. user key/username, to be assigned.
     */
    @Nonnull
    public String getTarget() {
        return target;
    }

    /**
     * Context to report progress to.
     * <p>
     * Proper implementation is only available during the execution of {@link UserKeyChangeHandler#update(Object)} or {@link UserNameChangeHandler#update(Object)}.
     * When trying to access it in other methods, the behavior of the returned object is unspecified.
     */
    @Override
    @Nonnull
    public Context getContext() {
        return context;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserPropertyChangeParameter that = (UserPropertyChangeParameter) o;
        return Objects.equals(original, that.original) &&
                Objects.equals(target, that.target) &&
                Objects.equals(context, that.context);
    }

    @Override
    public int hashCode() {
        return Objects.hash(original, target, context);
    }

    @Override
    public String toString() {
        return "UserPropertyChangeParameter{" +
                "original='" + original + '\'' +
                ", target='" + target + '\'' +
                '}';
    }
}
