package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalSpi;
import com.atlassian.jira.task.context.Context;
import com.atlassian.jira.task.context.Contexts;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Contains identifiers of the user to be anonymized.
 * Also contains {@code context} to report progress to.
 */
@ExperimentalSpi
public class UserAnonymizationParameter implements TaskContextAware {
    private final String userKey;
    private final Context context;

    public UserAnonymizationParameter(@Nonnull String userKey, @Nonnull Context context) {
        this.userKey = requireNonNull(userKey);
        this.context = requireNonNull(context);
    }

    public UserAnonymizationParameter(@Nonnull String userKey) {
        this(userKey, Contexts.nullContext());
    }

    /**
     * The user key of the user to be anonymized.
     */
    @Nonnull
    public String getUserKey() {
        return userKey;
    }

    /**
     * Context to report progress to.
     * <p>
     * Proper implementation is only available during the execution of {@link UserAnonymizationHandler#update(Object)}.
     * When trying to access it in other methods, the behavior of the returned object is unspecified.
     */
    @Override
    @Nonnull
    public Context getContext() {
        return context;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserAnonymizationParameter that = (UserAnonymizationParameter) o;
        return Objects.equals(userKey, that.userKey) &&
                Objects.equals(context, that.context);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userKey, context);
    }

    @Override
    public String toString() {
        return "UserAnonymizationParameter{" +
                "userKey='" + userKey + '\'' +
                '}';
    }
}
