package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalSpi;
import com.atlassian.jira.task.context.Context;
import com.atlassian.jira.task.context.Contexts;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Contains user keys of the user currently owning an entity, and of the user that the ownership is to be transferred to.
 * Also contains {@code context} to report progress to.
 */
@ExperimentalSpi
public class OwnershipTransferParameter implements TaskContextAware {
    private final String currentUserKey;
    private final String transferToUserKey;
    private final Context context;

    public OwnershipTransferParameter(@Nonnull String currentUserKey, @Nonnull String transferToUserKey, @Nonnull Context context) {
        this.currentUserKey = requireNonNull(currentUserKey);
        this.transferToUserKey = requireNonNull(transferToUserKey);
        this.context = requireNonNull(context);
    }

    /**
     * User key of the current entity owner to be updated.
     */
    @Nonnull
    public String getCurrentUserKey() {
        return currentUserKey;
    }

    /**
     * User key of the user that the ownership is to be transferred to.
     */
    @Nonnull
    public String getTransferToUserKey() {
        return transferToUserKey;
    }

    /**
     * Context to report progress to.
     * <p>
     * Proper implementation is only available during the execution of {@link OwnershipTransferHandler#update(Object)}.
     * When trying to access it in other methods, the behavior of the returned object is unspecified.
     */
    @Override
    @Nonnull
    public Context getContext() {
        return context;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        OwnershipTransferParameter that = (OwnershipTransferParameter) o;
        return Objects.equals(currentUserKey, that.currentUserKey) &&
                Objects.equals(transferToUserKey, that.transferToUserKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(currentUserKey, transferToUserKey);
    }

    @Override
    public String toString() {
        return "OwnershipTransferParameter{" +
                "currentUserKey='" + currentUserKey + '\'' +
                ", transferToUserKey='" + transferToUserKey + '\'' +
                '}';
    }
}
