package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalSpi;

import javax.annotation.Nonnull;
import java.util.Collection;

/**
 * Transfers ownership of entities before a user is disabled, to make sure the system keeps working properly. For example,
 * changes project lead and component lead, as having disabled users there can break the default assignee option.
 * <p>
 * This extension point is executed as the first step of the user anonymization process.
 * Any errors returned by the implementations will stop the process.
 * <p>
 * To register an instance of this handler to be executed by Jira, implement this interface and add
 * an {@code ownership-transfer-handler} module to the atlassian-plugin.xml, e.g.:
 * <p>
 * {@code <ownership-transfer-handler key="my-ownership-transfer-handler" class="com.example.OwnershipTransferHandlerImpl"
 * name="My ownership transfer handler" application="jira" />}
 */
@ExperimentalSpi
public interface OwnershipTransferHandler extends AnonymizationHandler<OwnershipTransferParameter> {

    /**
     * Used to display in the UI a list of entities that require an ownership transfer, without actually performing
     * the operation yet. Its goal is to inform the end-user about the scope of the changes.
     * Implementation should not use {@link OwnershipTransferParameter#getTransferToUserKey()} during computing affected entities.
     * <p>
     * This method should return results as fast as possible, as it is affecting the UI.
     *
     * @param ownershipTransferParameter stores original and target user keys
     * @return a collection of entities that will be affected by the ownership transfer
     */
    @Override
    @Nonnull
    Collection<AffectedEntity> getAffectedEntities(@Nonnull OwnershipTransferParameter ownershipTransferParameter);
}
