package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.bc.ServiceResult;

import javax.annotation.Nonnull;
import java.util.Collection;

/**
 * Performs operations related to anonymization process, e.g. anonymize user data in profile,
 * anonymize user key and referencing foreign keys so they don't contain personal data etc.
 *
 * @see UserAnonymizationHandler
 * @see OwnershipTransferHandler
 * @see UserKeyChangeHandler
 * @see UserNameChangeHandler
 * @since 8.3
 */
@ExperimentalApi
public interface AnonymizationHandler<T> {
    /**
     * Used to display in the UI a list of entities that are affected by anonymization process, without actually performing
     * the operation yet. Its goal is to inform the end-user about the scope of the changes.
     * <p>
     * This method should return results as fast as possible, as it is affecting the UI.
     *
     * @param parameter Parameter
     * @return a collection of affected entities
     */
    @Nonnull
    Collection<AffectedEntity> getAffectedEntities(@Nonnull T parameter);

    /**
     * Performs the update operations related to the affected entities.
     * <p>
     * This method is executed in a background task and thus its accuracy is more important than performance.
     * <p>
     * This method should be idempotent - multiple invocations of this method should lead to the same result.
     * In other words when method fails we can call it again and finish operation.
     * <p>
     * During execution, the implementation class will update the progress by typically calling
     * {@code Context.start(object);} on the context passed in {@code parameter}, and subsequently call
     * {@code Context.Task.complete();} on the the task object to indicate task has finished.
     * Implementation can also call {@code Context.setName("Name of current step")} to describe the task that is currently processed.
     *
     * @param parameter Parameter
     * @return a service outcome of the update operation, containing any errors
     */
    @Nonnull
    ServiceResult update(@Nonnull T parameter);

    /**
     * Returns number of tasks/stages to be executed as part of this handler execution. The number returned here must be
     * consistent with the number of updates performed as part of {@link #update(Object)}.
     * <p>
     * It's used to calculate task progress. Eg. handler returning 1000 from this method is responsible to execute 1000
     * calls to {@code Context.start(object);} and {@code Context.Task.complete();} in {@link #update(Object)}.
     * <p>
     * Default implementation returns 1000 tasks which means that each handle by default will have 1000 steps to complete, it's up
     * to handler to decide how progress is reported eg. complete all 1000 steps at once or in some chunks.
     * <p>
     * When visualizing progress each handler has progress bar part that has length proportional to the number returned by this method.
     * If you know that your handler will execute swiftly you can return lower number here eg. 100
     * (progress bar part for this handler will be 10 times shorter than those that return 1000).
     *
     * @return Number of tasks/stages to be executed
     */
    default int getNumberOfTasks(@Nonnull T userPropertyChangeParameter) {
        return 1000;
    }
}
