package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalSpi;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import javax.annotation.Nonnull;
import java.util.Objects;
import java.util.Optional;

/**
 * Represents an entity affected by a user property change or anonymization to be shown in the UI.
 */
@ExperimentalSpi
public class AffectedEntity {
    private final AffectedEntityType type;
    private final String descriptionKey;
    private final Optional<Long> numberOfOccurrences;
    private final Optional<AffectedEntityLink> link;

    /**
     * Use {@link AffectedEntity#newBuilder(AffectedEntityType)} to create instances of this class.
     */
    private AffectedEntity(@Nonnull AffectedEntityType type, @Nonnull String descriptionKey,
                           Long numberOfOccurrences, AffectedEntityLink link) {
        this.type = Objects.requireNonNull(type);
        this.descriptionKey = Objects.requireNonNull(descriptionKey);
        this.numberOfOccurrences = Optional.ofNullable(numberOfOccurrences);
        this.link = Optional.ofNullable(link);
    }

    /**
     * Describes the way in which the entity will be updated.
     */
    @Nonnull
    public AffectedEntityType getType() {
        return type;
    }

    /**
     * The i18n key used to display the name of the entity or the description of the operation.
     */
    @Nonnull
    public String getDescriptionKey() {
        return descriptionKey;
    }

    /**
     * The number of entities that will be updated.
     * Should be left empty if computing this number might take a long time.
     */
    @Nonnull
    public Optional<Long> getNumberOfOccurrences() {
        return numberOfOccurrences;
    }

    /**
     * A link allowing to find the affected data.
     */
    @Nonnull
    public Optional<AffectedEntityLink> getLink() {
        return link;
    }

    public static Builder newBuilder(@Nonnull AffectedEntityType type) {
        return new Builder(type);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AffectedEntity that = (AffectedEntity) o;

        return new EqualsBuilder()
                .append(type, that.type)
                .append(descriptionKey, that.descriptionKey)
                .append(numberOfOccurrences, that.numberOfOccurrences)
                .append(link, that.link)
                .isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37)
                .append(type)
                .append(descriptionKey)
                .append(numberOfOccurrences)
                .append(link)
                .toHashCode();
    }

    @Override
    public String toString() {
        return "AffectedEntity{" +
                "type=" + type +
                ", descriptionKey='" + descriptionKey + '\'' +
                ", numberOfOccurrences=" + numberOfOccurrences +
                ", link=" + link +
                '}';
    }

    public static class Builder {
        private AffectedEntityType type;
        private String descriptionKey;
        private Long numberOfOccurrences;
        private AffectedEntityLink link;

        private Builder(@Nonnull AffectedEntityType type) {
            this.type = Objects.requireNonNull(type);
        }

        public Builder descriptionKey(@Nonnull String descriptionKey) {
            this.descriptionKey = Objects.requireNonNull(descriptionKey);
            return this;
        }

        public Builder numberOfOccurrences(Long numberOfOccurrences) {
            this.numberOfOccurrences = numberOfOccurrences;
            return this;
        }

        public Builder link(AffectedEntityLink link) {
            this.link = link;
            return this;
        }

        public AffectedEntity build() {
            return new AffectedEntity(type, descriptionKey, numberOfOccurrences, link);
        }
    }
}
