package com.atlassian.jira.sharing;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.sharing.rights.ShareRight;
import com.atlassian.jira.sharing.rights.ShareRights;
import com.atlassian.jira.sharing.type.ShareType;

/**
 * Represents a JIRA share from the database.
 *
 * @since v3.13
 */
public interface SharePermission {
    /**
     * The identifier of the share. Can be null if the share has not been stored.
     *
     * @return the identifier of the share.
     */
    Long getId();

    /**
     * The type of the share (e.g. group). Should never be null.
     *
     * @return the type of the share.
     */
    ShareType.Name getType();

    /**
     * The first parameter of the permission. Its value can be null.
     *
     * @return the value of the permissions first parameter.
     */
    String getParam1();

    /**
     * The second parameter of the permission. Its value can be null. This parameter cannot have a value when {#getParam1} returns null.
     *
     * @return the value of the permissions second parameter.
     */
    String getParam2();

    /**
     * The rights for the permission. Can store value if user is allowed to 'view' or 'view and edit'
     *
     * @return the value of the share rights.
     *
     * @since v7.12
     */
    @ExperimentalApi
    default ShareRight getRights() {
        return ShareRights.VIEW;
    }

    /**
     * Checks if share permission has view right.
     *
     * @return true if this share permission has view right
     *
     * @since v7.12
     */
    @ExperimentalApi
    default boolean isView() {
        return getRights().hasViewRightsGranted();
    }

    /**
     * Checks if share permission has view and edit right.
     *
     * @return true if this share permission has view and edit right
     *
     * @since v7.12
     */
    @ExperimentalApi
    default boolean isEdit() {
        return getRights().hasEditRightsGranted();
    }
}
