package com.atlassian.jira.jql.query;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.jql.operand.QueryLiteral;
import com.atlassian.jira.jql.resolver.IndexInfoResolver;
import com.atlassian.jira.project.version.Version;
import com.atlassian.query.operator.Operator;
import com.google.common.collect.Lists;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * A factory for creating a Query for the {@link Operator#LIKE equals operator} for Version fields.
 *
 * @since v7.9
 */
@PublicApi
public class VersionLikeQueryFactory extends AbstractOperatorQueryFactory<Version> implements OperatorSpecificQueryFactory {
    private static final Logger log = LoggerFactory.getLogger(VersionLikeQueryFactory.class);

    VersionLikeQueryFactory(IndexInfoResolver indexInfoResolver) {
        super(indexInfoResolver);
    }

    public QueryFactoryResult createQueryForSingleValue(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        if (rawValues == null) {
            return QueryFactoryResult.createFalseResult();
        }
        if (Operator.LIKE.equals(operator)) {
            return handleLike(fieldName, getIndexValues(rawValues));
        } else if (Operator.NOT_LIKE.equals(operator)) {
            return handleNotLike(fieldName, getIndexValues(rawValues));
        } else {
            log.debug("Create query for single value was called with operator '{}', this only handles '~' and '!~'.", operator.getDisplayString());
            return QueryFactoryResult.createFalseResult();
        }
    }

    private QueryFactoryResult handleNotLike(final String fieldName, final List<String> indexValues) {
        final List<Query> notQueries = Lists.newArrayListWithCapacity(indexValues.size());
        for (String indexValue : indexValues) {
            notQueries.add(getTermQuery(fieldName, indexValue));
        }
        if (notQueries.isEmpty()) {
            return new QueryFactoryResult(TermQueryFactory.nonEmptyQuery(fieldName));
        } else {
            BooleanQuery.Builder boolQuery = new BooleanQuery.Builder();
            for (Query query : notQueries) {
                boolQuery.add(query, BooleanClause.Occur.MUST_NOT);
            }
            boolQuery.add(TermQueryFactory.nonEmptyQuery(fieldName), BooleanClause.Occur.FILTER);
            boolQuery.add(TermQueryFactory.visibilityQuery(fieldName), BooleanClause.Occur.FILTER);
            return new QueryFactoryResult(boolQuery.build(), false);
        }
    }


    private QueryFactoryResult handleLike(final String fieldName, final List<String> indexValues) {
        BooleanQuery.Builder orQuery = new BooleanQuery.Builder();
        for (String id : indexValues) {
            orQuery.add(getTermQuery(fieldName, id), BooleanClause.Occur.SHOULD);
        }
        return new QueryFactoryResult(orQuery.build());
    }

    public QueryFactoryResult createQueryForEmptyOperand(final String fieldName, final Operator operator) {
        if (operator == Operator.LIKE) {
            final QueryFactoryResult result = new QueryFactoryResult(TermQueryFactory.nonEmptyQuery(fieldName), true);
            return QueryFactoryResult.wrapWithVisibilityQuery(fieldName, result);
        } else if (operator == Operator.NOT_LIKE) {
            return new QueryFactoryResult(TermQueryFactory.nonEmptyQuery(fieldName));
        } else {
            log.debug("Create query for empty operand was called with operator '{}', this only handles '~' and '~!' ", operator.getDisplayString());
            return QueryFactoryResult.createFalseResult();
        }
    }

    public QueryFactoryResult createQueryForMultipleValues(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        log.debug("LIKE clauses do not support multi value operands.");
        return QueryFactoryResult.createFalseResult();
    }

    public boolean handlesOperator(final Operator operator) {
        return Operator.LIKE.equals(operator) || Operator.NOT_LIKE.equals(operator);
    }

}
