package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.DocumentConstants;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.SortedDocValuesField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.util.BytesRef;

import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * A custom field indexer for text fields that can be sorted
 *
 * @since v4.0
 */
public class ExactTextCustomFieldIndexer extends AbstractCustomFieldIndexer {
    private final CustomField field;

    public ExactTextCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager, final CustomField customField) {
        super(fieldVisibilityManager, notNull("field", customField));
        this.field = customField;
    }

    public void addDocumentFieldsSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, true);
    }

    public void addDocumentFieldsNotSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, false);
    }

    private void addDocumentFields(final Document doc, final Issue issue, final boolean searchable) {
        Object value = field.getValue(issue);
        final String stringValue = field.getCustomFieldType().getStringFromSingularObject(value);
        if (value != null) {
            if (searchable) {
                doc.add(new StringField(getDocumentFieldId(), stringValue, Field.Store.YES));
                doc.add(new SortedDocValuesField(getDocumentFieldId(), new BytesRef(stringValue)));
            } else {
                doc.add(new StoredField(getDocumentFieldId(), stringValue));
            }
        }
        final String string = FieldIndexerUtil.getValueForSorting((String) value);
        doc.add(new SortedDocValuesField(DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId(), new BytesRef(string)));
    }
}