package com.atlassian.jira.index.summary;

import com.atlassian.annotations.ExperimentalApi;
import java.util.Optional;

/**
 * Summary of the index replication queue, which contains data needed to diagnose the index replication status.
 *
 * @since v7.2.10
 */
@ExperimentalApi
public class IndexReplicationQueueSummary {

    private final String sendingNodeId;

    private final IndexReplicationQueueEntry lastConsumedOperation;

    private final IndexReplicationQueueEntry lastOperationInQueue;

    private final long queueSize;

    /**
     * Constructor with all object fields.
     *
     * @param sendingNodeId id of node that sent the replication request
     * @param lastConsumedOperation index replication operation last consumed from queue
     * @param lastOperationInQueue index replication operation last in queue
     * @param queueSize size of index replication queue
     */
    public IndexReplicationQueueSummary(final String sendingNodeId,
                                        final IndexReplicationQueueEntry lastConsumedOperation,
                                        final IndexReplicationQueueEntry lastOperationInQueue,
                                        final Long queueSize) {
        this.sendingNodeId = sendingNodeId;
        this.lastConsumedOperation = lastConsumedOperation;
        this.lastOperationInQueue = lastOperationInQueue;
        this.queueSize = queueSize;
    }

    /**
     * Returns the ID of a node whose replication operations are in the queue.
     *
     * @return id of node that sent the replication request
     */
    public String getSendingNodeId() {
        return sendingNodeId;
    }

    /**
     * Returns the index replication operation last consumed from the queue.
     * Optional can be empty if no operation was consumed from this queue yet.
     *
     * @return index replication operation last consumed from queue
     */
    public Optional<IndexReplicationQueueEntry> getLastConsumedOperation() {
        return Optional.ofNullable(lastConsumedOperation);
    }

    /**
     * Returns index replication operation last in queue.
     * Optional can be empty if queue is empty.
     *
     * @return index replication operation last in queue
     */
    public Optional<IndexReplicationQueueEntry> getLastOperationInQueue() {
        return Optional.ofNullable(lastOperationInQueue);
    }

    /**
     * Returns size of index replication queue.
     *
     * @return size of index replication queue
     */
    public long getQueueSize() {
        return queueSize;
    }
}
