package com.atlassian.jira.event.user.anonymize;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.annotations.ExperimentalApi;

import javax.annotation.Nonnull;
import java.util.Objects;

/**
 * This event is sent when a user key is changed.
 *
 * @since v8.3
 */
@ExperimentalApi
@EventName("jira.user.key.changed")
public class UserKeyChangedEvent {
    private final String oldUserKey;
    private final String newUserKey;

    /**
     * @param oldUserKey old user key (before the change)
     * @param newUserKey new user key (after the change)
     */
    public UserKeyChangedEvent(@Nonnull String oldUserKey, @Nonnull String newUserKey) {
        this.oldUserKey = Objects.requireNonNull(oldUserKey);
        this.newUserKey = Objects.requireNonNull(newUserKey);
    }

    /**
     * Returns old user key (before the change).
     *
     * @return old user key (before the change).
     */
    @Nonnull
    public String getOldUserKey() {
        return oldUserKey;
    }

    /**
     * Returns new user key (after the change).
     *
     * @return new user key (after the change).
     */
    @Nonnull
    public String getNewUserKey() {
        return newUserKey;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserKeyChangedEvent that = (UserKeyChangedEvent) o;
        return Objects.equals(oldUserKey, that.oldUserKey) &&
                Objects.equals(newUserKey, that.newUserKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(oldUserKey, newUserKey);
    }
}
