package com.atlassian.jira.event.user.anonymize;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.annotations.ExperimentalApi;

import java.util.Objects;

/**
 * This event is sent when a user has been anonymized.
 *
 * @since v8.4
 */
@ExperimentalApi
@EventName("jira.user.anonymized")
public class UserAnonymizationFinishedEvent {
    private final String userKey;
    private final String userName;
    private final String oldUserKey;
    private final String oldUserName;

    public UserAnonymizationFinishedEvent(String userKey, String userName, String oldUserKey, String oldUserName) {
        this.userKey = userKey;
        this.userName = userName;
        this.oldUserKey = oldUserKey;
        this.oldUserName = oldUserName;
    }

    /**
     * User key after anonymization process (current value).
     *
     * @return User key after anonymization process.
     */
    public String getUserKey() {
        return userKey;
    }

    /**
     * Username after anonymization process (current value).
     *
     * @return Username after anonymization process.
     */
    public String getUserName() {
        return userName;
    }

    /**
     * User key before anonymization process.
     *
     * @return User key before anonymization process.
     */
    public String getOldUserKey() {
        return oldUserKey;
    }

    /**
     * Username before anonymization process.
     *
     * @return Username before anonymization process.
     */
    public String getOldUserName() {
        return oldUserName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserAnonymizationFinishedEvent that = (UserAnonymizationFinishedEvent) o;
        return Objects.equals(userKey, that.userKey) &&
                Objects.equals(userName, that.userName) &&
                Objects.equals(oldUserKey, that.oldUserKey) &&
                Objects.equals(oldUserName, that.oldUserName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userKey, userName, oldUserKey, oldUserName);
    }

    @Override
    public String toString() {
        return "UserAnonymizationFinishedEvent{" +
                "userKey='" + userKey + '\'' +
                ", userName='" + userName + '\'' +
                ", oldUserKey='" + oldUserKey + '\'' +
                ", oldUserName='" + oldUserName + '\'' +
                '}';
    }
}
