package com.atlassian.jira.database;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.apache.commons.lang3.math.NumberUtils.toInt;

/**
 * An enum representing the supported database vendors that JIRA can connect to.
 *
 * @since v7.0
 */
public enum DatabaseVendor {
    POSTGRES("Postgres"),
    ORACLE("Oracle") {
        @Override
        public String getVersion(final String version) {
            return getOracleDatabaseVersion(version);
        }
    },
    SQL_SERVER("SQL Server") {
        @Override
        public String getVersion(final String version) {
            return getSQLServerVersion(version);
        }
    },
    H2("H2"),
    MY_SQL("MySQL"),
    FAKE_DATABASE_FOR_TESTING("fakedatabase");

    public String getHumanReadableName() {
        return humanReadableName;
    }

    final String humanReadableName;


    DatabaseVendor(final String humanReadableName){
        this.humanReadableName = humanReadableName;
    }

    private static final Pattern VERSION_PATTERN = Pattern.compile("(\\d+(\\.\\d+)*).*");

    private static final Pattern ORACLE_MAJOR_MINOR_PATTERN = Pattern.compile("^.*?(\\d+\\.\\d+)\\.\\d+\\.\\d+.*?", Pattern.DOTALL);

    private static final Map<String, String> ORACLE_VERSION_ALIASES =
            ImmutableMap.<String, String>builder()
                    .put("19.0", "19C")
                    .put("18.0", "18C")
                    .put("12.2", "12C R2")
                    .put("12.1", "12C R1")
                    .put("11.2", "11G R2")
                    .put("11.1", "11G R1").build();

    private static final Map<String, String> MSSQL_VERSION_ALIASES =
            ImmutableMap.<String, String>builder()
                    .put("14.*", "2017")
                    .put("13.*", "2016")

                    //This is also the database version Azure pretends to be by default when I tested it  on 14-05-2018
                    //It's possible for users to set it to be a different version however
                    //https://azure.microsoft.com/en-us/blog/default-compatibility-level-140-for-azure-sql-databases/
                    .put("12.*", "2014")

                    .put("11.*", "2012")
                    .put("10.5.+", "2008 R2")
                    .put("10.[0-4].*", "2008")
                    .put("9.*", "2005")
                    .put("8.*", "2000").build();

    /**
     * Oracle markets their databases under different version names than their database reports.
     * This here translates Oracle versions so we can show error messages referencing the version they're familiar with (the marketing version)
     * <p>
     * Takes a Oracle Database version string and gets its corresponding marketing version name
     *
     * @param version The version as a String, e.g.: "Oracle Database 12c Standard Edition Release 12.2.0.1.0 - 64bit Production".
     * @return marketing version name e.g "12c R1"
     */
    private static String getOracleDatabaseVersion(final String version) {
        final Matcher m = ORACLE_MAJOR_MINOR_PATTERN.matcher(version);
        if (!m.matches()) {
            throw new IllegalArgumentException("the version string retrieved from the database does not seem to contain a version number" +
                    " (version string: \""+ StringEscapeUtils.escapeJava(version)+"\")");
        }
        return ORACLE_VERSION_ALIASES
                .entrySet()
                .stream()
                .filter(entry -> m.group(1).equals(entry.getKey()))
                .map(Map.Entry::getValue)
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("the version number retrieved from the database isn't a known version of Oracle Database" +
                        " (version string: \""+ StringEscapeUtils.escapeJava(version)+"\")"));
    }

    /**
     * Microsoft SQL server markets their databases under different version names than their database reports their version
     * So this is so we can show error messages referencing the version they're familiar with (the marketing version)
     * <p>
     * Takes a Microsoft SQL Server version string and gets its corresponding marketing version name
     *
     * @param version The version as a String, e.g.: "10.50.1600.1".
     * @return marketing version name e.g "2008 R2"
     */
    private static String getSQLServerVersion(final String version) {
        return MSSQL_VERSION_ALIASES
                .entrySet()
                .stream()
                .filter(entry -> version.matches(entry.getKey()))
                .map(Map.Entry::getValue)
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("the version number retrieved from the database isn't a known version of SQL Server" +
                        " (version string: \""+ StringEscapeUtils.escapeJava(version)+"\")"));
    }

    /**
     * This is a public static delegate to private static method getSQLServerVersion().
     * It's there to satisfy API checker / backwards compatibility. Deprecated since Jira 8.3
     * @param version The version as a String, e.g.: "10.50.1600.1".
     * @return marketing version name e.g "2008 R2"
     */
    @Deprecated
    public static String getSQLServerVersionIfExists(final String version) {
        try {
            return getSQLServerVersion(version);
        } catch (final IllegalArgumentException e) {
            return version;
        }
    }

    /**
     * Gets a release version of a database, i.e.:
     * - minor.minor (without micro/bugfix position)
     * or
     * - marketing name for a release
     * Note: this method is overridden for some vendors
     * @param version The version string as retrieved from the DatabaseAccessor
     * @return release version
     * @throws IllegalArgumentException in case of failure, message explains the reason
     */
    public String getVersion(final String version) {
        final Matcher matcher = VERSION_PATTERN.matcher(version);
        if (matcher.matches()) {
            String[] parts = StringUtils.split(matcher.group(1), ".");
            final int major = parts.length > 0 ? toInt(parts[0]) : 0;
            final int minor = parts.length > 1 ? toInt(parts[1]) : 0;
            return major + "." + minor;
        }
        throw new IllegalArgumentException("the version number retrieved from the database isn't a version number we expected to get (major.minor.micro)" +
                " (version string: \""+ StringEscapeUtils.escapeJava(version)+"\")");
    }

    /**
     * Gets a release version of a database like getVersion() method, but does not throw in case of failure.
     * @param version The version string as retrieved from the DatabaseAccessor
     * @return release version or input version in case of failure
     */
    public String getHumanReadableVersion(final String version) {
        try {
            return getVersion(version);
        } catch (final IllegalArgumentException e) {
            return version;
        }
    }
}