package com.atlassian.jira.config;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.bc.ServiceOutcome;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.search.SearchException;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;
import org.apache.commons.lang.StringUtils;

import java.util.Collection;
import java.util.List;

/**
 * Service for managing issue type schemes and their project associations.
 * <p>
 *     Note, this service does not support issue type migrations. Issue type scheme updates and project associations that
 *     require migrations will fail and return with error outcomes.
 * </p>
 *
 * @since 8.0
 */
@PublicApi
public interface IssueTypeSchemeService {

    /**
     * Creates a new issue type scheme with the specified name, description, issue types, and default issue type. Newly
     * created issue type schemes do not start out associated with any projects.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the name is null or an empty string</li>
     *         <li>no issue type ids are specified</li>
     *         <li>the new scheme's default issue type is not listed among the <code>issueTypeIds</code></li>
     *     </ul>
     * </p>
     * @param user user that performs the create operation. Must be a Jira administrator.
     * @param name what the new issue type scheme will be called. Must not be null or empty.
     * @param description optional additional information about the new scheme
     * @param issueTypeIds which issue types will be available to projects associated with this scheme. Must not be null
     *                     or empty.
     * @param defaultIssueTypeId specifies which of the new scheme's issue types will be initially selected when creating
     *                           a new issue (in a project associated with this scheme). Can be null (meaning no default),
     *                           but otherwise must be an id found in <code>issueTypeIds</code>.
     * @return a service outcome containing the created issue type scheme or an error if something went wrong during the operation.
     */
    ServiceOutcome<FieldConfigScheme> createIssueTypeScheme(ApplicationUser user, String name, String description,
                                                            List<String> issueTypeIds, String defaultIssueTypeId);
    /**
     * Creates a new issue type scheme with the specified name, description, issue types, and an unspecified default issue
     * type. Newly created issue type schemes do not start out associated with any projects.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the name is null or an empty string</li>
     *         <li>no issue type ids are specified</li>
     *     </ul>
     * </p>
     * @param user user that performs the create operation. Must be a Jira administrator.
     * @param name what the new issue type scheme will be called. Must not be null or empty.
     * @param description optional additional information about the new scheme
     * @param issueTypeIds which issue types will be available to projects associated with this scheme. Must not be null
     *                     or empty.
     * @return a service outcome containing the created issue type scheme or an error if something went wrong during the operation.
     */
    default ServiceOutcome<FieldConfigScheme> createIssueTypeScheme(ApplicationUser user, String name, String description,
                                                                    List<String> issueTypeIds) {
        return createIssueTypeScheme(user, name, description, issueTypeIds, null);
    }

    /**
     * Retrieves and returns all issue type schemes in the Jira.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *     </ul>
     * </p>
     * @param user user that performs the retrieval operation. Must be a Jira administrator.
     * @return a service outcome containing the issue type schemes found in the Jira
     */
    ServiceOutcome<List<FieldConfigScheme>> getAllIssueTypeSchemes(ApplicationUser user);

    /**
     * Retrieves the issue type scheme specified by the <code>schemeId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the retrieval operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to retrieve.
     * @return a service outcome containing the specified issue type scheme, if it exists.
     */
    ServiceOutcome<FieldConfigScheme> getIssueTypeScheme(ApplicationUser user, long schemeId);

    /**
     * Retrieves the issue type scheme specified by the <code>schemeId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the retrieval operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to retrieve.
     * @return a service outcome containing the specified issue type scheme, if it exists.
     */
    default ServiceOutcome<FieldConfigScheme> getIssueTypeScheme(ApplicationUser user, String schemeId) {

        //We should revisit the re-use of this block of code. See @ https://bulldog.internal.atlassian.com/browse/DELTA-542
        if (StringUtils.isNumeric(schemeId)) {
            return getIssueTypeScheme(user, Long.parseLong(schemeId));
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);

    }

    /**
     * Updates the issue type scheme specified by <code>schemeId</code> to have the given name, description, issue types,
     * and default issue type. Project associations are not within the scope of this operation; use
     * {@link IssueTypeSchemeService#setProjectAssociations(ApplicationUser, long, List) setProjectAssociations} or
     * {@link IssueTypeSchemeService#addProjectAssociations(ApplicationUser, long, List) addProjectAssociations} instead.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>an attempt is made to update the system's default issue type scheme</li>
     *         <li>the name is null or an empty string</li>
     *         <li>no issue type ids are specified</li>
     *         <li>the new default issue type is not listed among the <code>issueTypeIds</code> being assigned</li>
     *         <li>changing the scheme's issue types requires a migration of issues by any of its associated projects</li>
     *     </ul>
     * </p>
     * @param user user that performs the update operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to update.
     * @param name what the issue type scheme will be called after the update. Must not be null or empty.
     * @param description optional additional information about the updated scheme
     * @param issueTypeIds which issue types will be available to projects associated with this scheme. Must not be null
     *                     or empty.
     * @param defaultIssueTypeId specifies which of the scheme's issue types will be initially selected when creating
     *                           a new issue (in a project associated with this scheme). Can be null (meaning no default),
     *                           but otherwise must be an id found in <code>issueTypeIds</code> being assigned.
     * @return a service outcome containing the updated issue type scheme or an error if something went wrong during the operation.
     */
    ServiceOutcome<FieldConfigScheme> updateIssueTypeScheme(ApplicationUser user, long schemeId, String name, String description,
                                                            List<String> issueTypeIds, String defaultIssueTypeId);

    /**
     * Updates the issue type scheme specified by <code>schemeId</code> to have the given name, description, issue types,
     * and default issue type. Project associations are not within the scope of this operation; use
     * {@link IssueTypeSchemeService#setProjectAssociations(ApplicationUser, long, List) setProjectAssociations} or
     * {@link IssueTypeSchemeService#addProjectAssociations(ApplicationUser, long, List) addProjectAssociations} instead.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>an attempt is made to update the system's default issue type scheme</li>
     *         <li>the name is null or an empty string</li>
     *         <li>no issue type ids are specified</li>
     *         <li>the new default issue type is not listed among the <code>issueTypeIds</code> being assigned</li>
     *         <li>changing the scheme's issue types requires a migration of issues by any of its associated projects</li>
     *     </ul>
     * </p>
     * @param user user that performs the update operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to update.
     * @param name what the issue type scheme will be called after the update. Must not be null or empty.
     * @param description optional additional information about the updated scheme
     * @param issueTypeIds which issue types will be available to projects associated with this scheme. Must not be null
     *                     or empty.
     * @param defaultIssueTypeId specifies which of the scheme's issue types will be initially selected when creating
     *                           a new issue (in a project associated with this scheme). Can be null (meaning no default),
     *                           but otherwise must be an id found in <code>issueTypeIds</code> being assigned.
     * @return a service outcome containing the updated issue type scheme or an error if something went wrong during the operation.
     */
    default ServiceOutcome<FieldConfigScheme> updateIssueTypeScheme(ApplicationUser user, String schemeId, String name, String description,
                                                            List<String> issueTypeIds, String defaultIssueTypeId) {
        if (StringUtils.isNumeric(schemeId)) {
            return updateIssueTypeScheme(user, Long.parseLong(schemeId), name, description, issueTypeIds, defaultIssueTypeId);
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }

    /**
     * Updates the issue type scheme specified by <code>schemeId</code> to have the given name, description, issue types,
     * and an unspecified default issue type. Project associations are not within the scope of this operation; use
     * {@link IssueTypeSchemeService#setProjectAssociations(ApplicationUser, long, List) setProjectAssociations} or
     * {@link IssueTypeSchemeService#addProjectAssociations(ApplicationUser, long, List) addProjectAssociations} instead.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>an attempt is made to update the system's default issue type scheme</li>
     *         <li>the name is null or an empty string</li>
     *         <li>no issue type ids are specified</li>
     *         <li>changing the scheme's issue types requires a migration of issues by any of its associated projects</li>
     *     </ul>
     * </p>
     * @param user user that performs the update operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to update.
     * @param name what the issue type scheme will be called after the update. Must not be null or empty.
     * @param description optional additional information about the updated scheme
     * @param issueTypeIds which issue types will be available to projects associated with this scheme. Must not be null
     *                     or empty.
     * @return a service outcome containing the updated issue type scheme or an error if something went wrong during the operation.
     */
    default ServiceOutcome<FieldConfigScheme> updateIssueTypeScheme(ApplicationUser user, long schemeId, String name, String description,
                                                            List<String> issueTypeIds) {
        return updateIssueTypeScheme(user, schemeId, name, description, issueTypeIds, null);
    }

    /**
     * Updates the issue type scheme specified by <code>schemeId</code> to have the given name, description, issue types,
     * and an unspecified default issue type. Project associations are not within the scope of this operation; use
     * {@link IssueTypeSchemeService#setProjectAssociations(ApplicationUser, long, List) setProjectAssociations} or
     * {@link IssueTypeSchemeService#addProjectAssociations(ApplicationUser, long, List) addProjectAssociations} instead.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>an attempt is made to update the system's default issue type scheme</li>
     *         <li>the name is null or an empty string</li>
     *         <li>no issue type ids are specified</li>
     *         <li>changing the scheme's issue types requires a migration of issues by any of its associated projects</li>
     *     </ul>
     * </p>
     * @param user user that performs the update operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to update.
     * @param name what the issue type scheme will be called after the update. Must not be null or empty.
     * @param description optional additional information about the updated scheme
     * @param issueTypeIds which issue types will be available to projects associated with this scheme. Must not be null
     *                     or empty.
     * @return a service outcome containing the updated issue type scheme or an error if something went wrong during the operation.
     */
    default ServiceOutcome<FieldConfigScheme> updateIssueTypeScheme(ApplicationUser user, String schemeId, String name, String description,
                                                                    List<String> issueTypeIds) {
        return updateIssueTypeScheme(user, schemeId, name, description, issueTypeIds, null);
    }

    /**
     * Deletes the issue type scheme specified by the <code>schemeId</code>. Any projects associated with this scheme will
     * revert to the global default issue type scheme.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>an attempt is made to delete the system's default issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the delete operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to delete.
     * @return an empty service outcome indicating a successful deletion or an error if something went wrong during the operation.
     */
    ServiceOutcome<Void> deleteIssueTypeScheme(ApplicationUser user, long schemeId);

    /**
     * Deletes the issue type scheme specified by the <code>schemeId</code>. Any projects associated with this scheme will
     * revert to the global default issue type scheme.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>an attempt is made to delete the system's default issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the delete operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to delete.
     * @return an empty service outcome indicating a successful deletion or an error if something went wrong during the operation.
     */
    default ServiceOutcome<Void> deleteIssueTypeScheme(ApplicationUser user, String schemeId) {
        if (StringUtils.isNumeric(schemeId)) {
            return deleteIssueTypeScheme(user, Long.parseLong(schemeId));
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }

    /**
     * Adds additional project-scheme associations between the issue type scheme specified by the <code>schemeId</code>
     * and the projects given by the <code>projectIdsOrKeys</code> collection. This operation does not modify or remove
     * any existing project associations that the scheme has.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>any of the <code>projectIdsOrKeys</code> does not reference a valid project</li>
     *         <li>associating the scheme with any of the specified projects requires an issue migration</li>
     *     </ul>
     * </p>
     * @param user user that performs the association operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to add project associations to.
     * @param projectIdsOrKeys lookup keys/ids for projects to be associated with the issue type scheme
     * @return an empty service outcome indicating the successful addition of associated projects or an error if something
     * went wrong during the operation.
     */
    ServiceOutcome<Void> addProjectAssociations(ApplicationUser user, long schemeId, List<String> projectIdsOrKeys);

    /**
     * Adds additional project-scheme associations between the issue type scheme specified by the <code>schemeId</code>
     * and the projects given by the <code>projectIdsOrKeys</code> collection. This operation does not modify or remove
     * any existing project associations that the scheme has.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>any of the <code>projectIdsOrKeys</code> does not reference a valid project</li>
     *         <li>associating the scheme with any of the specified projects requires an issue migration</li>
     *     </ul>
     * </p>
     * @param user user that performs the association operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme to add project associations to.
     * @param projectIdsOrKeys lookup keys/ids for projects to be associated with the issue type scheme
     * @return an empty service outcome indicating the successful addition of associated projects or an error if something
     * went wrong during the operation.
     */
    default ServiceOutcome<Void> addProjectAssociations(ApplicationUser user, String schemeId, List<String> projectIdsOrKeys) {

        if (StringUtils.isNumeric(schemeId)) {
            return addProjectAssociations(user, Long.parseLong(schemeId), projectIdsOrKeys);
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }

    /**
     * Retrieves the project-scheme associations for the issue type scheme specified by the <code>schemeId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the association operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme
     * @return the projects associated with the specified <code>schemeId</code> or an error if something went wrong
     * during the operation.
     */
    ServiceOutcome<List<Project>> getAssociatedProjects(ApplicationUser user, long schemeId);

    /**
     * Retrieves the project-scheme associations for the issue type scheme specified by the <code>schemeId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the association operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme
     * @return the projects associated with the specified <code>schemeId</code> or an error if something went wrong
     * during the operation.
     */
    default ServiceOutcome<List<Project>> getAssociatedProjects(ApplicationUser user, String schemeId) {

        if (StringUtils.isNumeric(schemeId)) {
            return getAssociatedProjects(user, Long.parseLong(schemeId));
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }



    /**
     * Replaces the project-scheme associations for the issue type scheme specified by the <code>schemeId</code>
     * with the projects given by the <code>projectIdsOrKeys</code> collection. Again, this operation overwrites any
     * existing project associations that the scheme has.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>any of the <code>projectIdsOrKeys</code> does not reference a valid project</li>
     *         <li>associating the scheme with any of the specified projects requires an issue migration</li>
     *     </ul>
     * </p>
     * @param user user that performs the association operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme.
     * @param projectIdsOrKeys lookup keys/ids for projects to be associated with the issue type scheme
     * @return an empty service outcome indicating the successful replacement of associated projects or an error if
     * something went wrong during the operation.
     */
    ServiceOutcome<Void> setProjectAssociations(ApplicationUser user, long schemeId, List<String> projectIdsOrKeys);

    /**
     * Replaces the project-scheme associations for the issue type scheme specified by the <code>schemeId</code>
     * with the projects given by the <code>projectIdsOrKeys</code> collection. Again, this operation overwrites any
     * existing project associations that the scheme has.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>any of the <code>projectIdsOrKeys</code> does not reference a valid project</li>
     *         <li>associating the scheme with any of the specified projects requires an issue migration</li>
     *     </ul>
     * </p>
     * @param user user that performs the association operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme.
     * @param projectIdsOrKeys lookup keys/ids for projects to be associated with the issue type scheme
     * @return an empty service outcome indicating the successful replacement of associated projects or an error if
     * something went wrong during the operation.
     */
    default ServiceOutcome<Void> setProjectAssociations(ApplicationUser user, String schemeId, List<String> projectIdsOrKeys) {

        if (StringUtils.isNumeric(schemeId)) {
            return setProjectAssociations(user, Long.parseLong(schemeId), projectIdsOrKeys);
        }

        return errorCase("not a valid scheme id " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }

    /**
     * Removes all project associations for the issue type scheme specified by the <code>schemeId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>the schemeId corresponds with the default global issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the removal operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme.
     * @return an empty service outcome indicating the successful removal of the associated projects or an error if
     * something went wrong during the operation.
     */
    ServiceOutcome<Void> removeAllProjectAssociations(ApplicationUser user, long schemeId);

    /**
     * Removes all project associations for the issue type scheme specified by the <code>schemeId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>the schemeId corresponds with the default global issue type scheme</li>
     *     </ul>
     * </p>
     * @param user user that performs the removal operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme.
     * @return an empty service outcome indicating the successful removal of the associated projects or an error if
     * something went wrong during the operation.
     */
    default ServiceOutcome<Void> removeAllProjectAssociations(ApplicationUser user, String schemeId) {

        if (StringUtils.isNumeric(schemeId)) {
            return removeAllProjectAssociations(user, Long.parseLong(schemeId));
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }

    /**
     * For the issue type scheme specified by the <code>schemeId</code>, removes the project association given by
     * <code>projKeyOrId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>the schemeId corresponds with the default global issue type scheme</li>
     *         <li>the projKeyOrId doesn't correspond with a project that the issue type scheme is associated with</li>
     *         <li>any of the <code>projectIdsOrKeys</code> does not reference a valid project</li>
     *     </ul>
     * </p>
     * @param user user that performs the removal operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme.
     * @param projKeyOrId key/id for the project that we're un-associating with the scheme
     * @return an empty service outcome indicating the successful removal of the associated project or an error if
     * something went wrong during the operation.
     */
    ServiceOutcome<Void> removeProjectAssociation(ApplicationUser user, long schemeId, String projKeyOrId);

    /**
     * For the issue type scheme specified by the <code>schemeId</code>, removes the project association given by
     * <code>projKeyOrId</code>.
     * <p>
     *     This operation will fail validation and return an unsuccessful outcome if:
     *     <ul>
     *         <li>the user is not a Jira administrator</li>
     *         <li>the schemeId does not correspond with a valid issue type scheme</li>
     *         <li>the schemeId corresponds with the default global issue type scheme</li>
     *         <li>the projKeyOrId doesn't correspond with a project that the issue type scheme is associated with</li>
     *         <li>any of the <code>projectIdsOrKeys</code> does not reference a valid project</li>
     *     </ul>
     * </p>
     * @param user user that performs the removal operation. Must be a Jira administrator.
     * @param schemeId unique id that references an existing issue type scheme.
     * @param projKeyOrId key/id for the project that we're un-associating with the scheme
     * @return an empty service outcome indicating the successful removal of the associated project or an error if
     * something went wrong during the operation.
     */
    default ServiceOutcome<Void> removeProjectAssociation(ApplicationUser user, String schemeId, String projKeyOrId) {

        if (StringUtils.isNumeric(schemeId)) {
            return removeProjectAssociation(user, Long.parseLong(schemeId), projKeyOrId);
        }

        return errorCase("not a valid scheme id: " + schemeId, ErrorCollection.Reason.VALIDATION_FAILED);
    }

    /**
     * Figures out which issues in the specified projects would require migration if only the <code>chosenIssueTypeIds</code>
     * were made available to the projects.
     * @param user user that performs the lookup operation.
     * @param chosenIssueTypeIds collection of issue type ids against which we're checking the need for issue migrations.
     * @param projectIds collection of unique identifiers for the projects whose issues may require migration.
     * @return a service outcome containing the list of issues requiring migration (if any) or an error if something went wrong during the operation.
     */
    ServiceOutcome<List<Issue>> getIssuesRequiringMigration(ApplicationUser user, final Collection<String> chosenIssueTypeIds, List<Long> projectIds);

    /**
     * Figures out which issues in the specified projects would require migration if the given <code>issueTypeScheme</code>
     * were to be associated with them.
     * @param user user that performs the lookup operation.
     * @param issueTypeScheme issue type scheme against which we're checking the need for issue migrations.
     * @param projectIds collection of unique identifiers for the projects whose issues may require migration.
     * @return a service outcome containing the list of issues requiring migration (if any) or an error if something went wrong during the operation.
     */
    ServiceOutcome<List<Issue>> getIssuesRequiringMigrationDueToSchemeAssociation(ApplicationUser user, FieldConfigScheme issueTypeScheme, List<Long> projectIds);

    /**
     * For the projects specified by <code>projectIds</code>, determines how many issues are of the types given by
     * <code>issueTypIds</code>. This functionality is provided as a separate hook, as it can be implemented more efficiently this
     * way rather than via {@link IssueTypeSchemeService#getIssuesMatchingTypesInProjects(ApplicationUser, Collection, Collection)
     * getIssuesMatchingTypesInProjects(..).size()}.
     *
     * @param user user that performs the lookup operation.
     * @param projectIds unique identifiers for the projects whose issues we'd like to inspect.
     * @param issueTypeIds unique identifiers for the issue types that we're looking for instances of in the projects.
     * @return the number of issues in the given projects that are of the types specified by <code>issueTypeIds</code>
     * @throws SearchException
     */
    long getNumIssuesMatchingTypesInProjects(ApplicationUser user, Collection<Long> projectIds, Collection<String> issueTypeIds) throws SearchException;

    /**
     * For the projects specified by <code>projectIds</code>, finds the issues that are of the types given by
     * <code>issueTypIds</code>. Note: rather than calling .size() on the results of this method, use the more efficient
     * {@link IssueTypeSchemeService#getNumIssuesMatchingTypesInProjects(ApplicationUser, Collection, Collection)
     * getNumIssuesMatchingTypesInProjects(...)} instead.
     *
     * @param user user that performs the lookup operation.
     * @param projectIds unique identifiers for the projects whose issues we'd like to inspect.
     * @param issueTypeIds unique identifiers for the issue types that we're looking for instances of in the projects.
     * @return List of Issues in the given projects that are of the types specified by <code>issueTypeIds</code>
     * @throws SearchException
     */
    List<Issue> getIssuesMatchingTypesInProjects(ApplicationUser user, Collection<Long> projectIds, Collection<String> issueTypeIds) throws SearchException;

    /**
     * Creates a failed service outcome with the specified error <code>msg</code> and <code>reason</code>.
     *
     * @param msg string explanation of the failure.
     * @param reason Motivation/reason behind the failure.
     * @param <T> generic type that corresponds with what the successful outcome would have been.
     * @return a failed service outcome with the specified error.
     */
    <T> ServiceOutcome<T> errorCase(String msg, ErrorCollection.Reason reason);

    /**
     * Looks up and returns the default issue type (the one that is presented as the initial/selected choice to users
     * when creating a new Issue, for instance) for the specified issue type scheme.
     * @param issueTypeScheme scheme whose default issue type we'd like to look up.
     * @return the default issue type for the specified issue type scheme--or null if it is unspecified.
     */
    IssueType getDefaultIssueType(FieldConfigScheme issueTypeScheme);
}
