package com.atlassian.jira.auditing;

import com.atlassian.annotations.ExperimentalApi;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nullable;
import java.util.List;

/**
 * This class represents filters that can be applied when getting Audit log records.
 * Use AuditingFilter#builder() to acquire builder for constructing object of this class.
 *
 * @since v6.3
 */
@ExperimentalApi
public class AuditingFilter {
    static public Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String filter;
        private Long fromTimestamp;
        private Long toTimestamp;
        private List<Long> projectIds;
        private List<String> userIds;
        private boolean hideExternalDirectories = false;

        private Builder() {
        }

        /**
         * @param filter text filter query; each record in result set will need to contain this text
         */
        public Builder filter(@Nullable final String filter) {
            this.filter = filter;
            return this;
        }

        /**
         * @param fromTimestamp timestamp in past to narrow the results set; fromTimestamp must be less or equal
         *                      toTimestamp, otherwise the result set will be empty only records that where created in the same moment or after
         *                      the fromTimestamp will be provided in results set
         */
        public Builder fromTimestamp(@Nullable final Long fromTimestamp) {
            this.fromTimestamp = fromTimestamp;
            return this;
        }

        /**
         * @param toTimestamp timestamp in past to narrow the results set; fromTimestamp must be less or equal
         *                    toTimestamp, otherwise the result set will be empty only records that where created in the same moment or earlier
         *                    than the toTimestamp will be provided in results set
         */
        public Builder toTimestamp(@Nullable final Long toTimestamp) {
            this.toTimestamp = toTimestamp;
            return this;
        }

        /**
         * @param projectIds The id of the project for which to return audit records.
         */
        public Builder projectIds(@Nullable final List<Long> projectIds) {
            this.projectIds = projectIds == null ? ImmutableList.of() : ImmutableList.copyOf(projectIds);
            return this;
        }

        /**
         * @param userIds The id of the users for which to return audit records.
         */
        public Builder userIds(@Nullable final List<String> userIds) {
            this.userIds = userIds == null ? ImmutableList.of() : ImmutableList.copyOf(userIds);
            return this;
        }

        /**
         * @param hideExternalDirectories if true only records related to internal directory will be returned
         */
        public Builder setHideExternalDirectories(boolean hideExternalDirectories) {
            this.hideExternalDirectories = hideExternalDirectories;
            return this;
        }

        public AuditingFilter build() {
            return new AuditingFilter(filter, fromTimestamp, toTimestamp, projectIds, userIds, hideExternalDirectories);
        }
    }

    private final String filter;
    private final Long fromTimestamp;
    private final Long toTimestamp;
    private final List<Long> projectIds;
    private final List<String> userIds;
    private final boolean hideExternalDirectories;

    private AuditingFilter(@Nullable String filter,
                           @Nullable Long fromTimestamp,
                           @Nullable Long toTimestamp,
                           @Nullable List<Long> projectIds,
                           @Nullable List<String> userIds,
                           boolean hideExternalDirectories) {
        this.filter = filter;
        this.fromTimestamp = fromTimestamp;
        this.toTimestamp = toTimestamp;
        this.projectIds = projectIds;
        this.userIds = userIds;
        this.hideExternalDirectories = hideExternalDirectories;
    }

    @Nullable
    public String getFilter() {
        return filter;
    }

    @Nullable
    public Long getFromTimestamp() {
        return fromTimestamp;
    }

    @Nullable
    public Long getToTimestamp() {
        return toTimestamp;
    }

    /**
     * @since v7.4
     * @return The id of the projects for which to return audit records
     */
    @Nullable
    public List<Long> getProjectIds() {
        return projectIds;
    }

    /**
     * @since v7.4
     * @return true when there are projects defined in filter
     */
    public Boolean hasProjectIds() {
        return projectIds != null && !projectIds.isEmpty();
    }

    /**
     * @since v7.4
     * @return The id of the users for which to return audit records
     */
    @Nullable
    public List<String> getUserIds() {
        return userIds;
    }

    /**
     * @since v7.4
     * @return true, when there are users defined in audit filter
     */
    public Boolean hasUserIds() {
        return userIds != null && !userIds.isEmpty();
    }

    public boolean isHideExternalDirectories() {
        return hideExternalDirectories;
    }
}
