package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Contains data necessary to validate and perform user anonymization.
 */
@ExperimentalApi
public class AnonymizationParameters implements Serializable {
    private final String userKey;
    private final boolean rerunPluginPoints;
    private final String oldUserKey;
    private final String oldUserName;
    private final String newOwnerKey;
    private final boolean getAffectedEntities;
    private final transient ApplicationUser executor;

    private AnonymizationParameters(@Nonnull String userKey,
                                    boolean rerunPluginPoints,
                                    @Nullable String oldUserKey,
                                    @Nullable String oldUserName,
                                    @Nullable String newOwnerKey,
                                    boolean getAffectedEntities,
                                    @Nonnull ApplicationUser executor) {
        this.userKey = requireNonNull(userKey);
        this.rerunPluginPoints = rerunPluginPoints;
        this.oldUserKey = oldUserKey;
        this.oldUserName = oldUserName;
        this.newOwnerKey = newOwnerKey;
        this.getAffectedEntities = getAffectedEntities;
        this.executor = requireNonNull(executor);
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * @return User key of user being anonymized.
     */
    @Nonnull
    public String getUserKey() {
        return userKey;
    }

    public boolean isRerunPluginPoints() {
        return rerunPluginPoints;
    }

    /**
     * @return New owner key for entities that need to have their ownership transferred.
     */
    @Nullable
    public String getNewOwnerKey() {
        return newOwnerKey;
    }

    /**
     * @return User who executes anonymization process.
     */
    @Nonnull
    public ApplicationUser getExecutor() {
        return executor;
    }

    /**
     * @return User key passed as {@link UserPropertyChangeParameter#getOriginal()} to user key change handler.
     * Used when retrying interrupted user key change process when user key has already been changed internally but not
     * all {@link UserKeyChangeHandler}s were executed.
     */
    @Nullable
    public String getOldUserKey() {
        return oldUserKey;
    }

    /**
     * @return User name passed as {@link UserPropertyChangeParameter#getOriginal()} to user name change handler.
     * Used when retrying interrupted user name change process when user name has already been changed internally but not
     * all {@link UserNameChangeHandler}s were executed.
     */
    @Nullable
    public String getOldUserName() {
        return oldUserName;
    }

    /**
     * Indicates whether affected entities should be used during validation phase.
     *
     * @return true if affected entities should be used (and returned) during validation phase, false otherwise.
     */
    public boolean isGetAffectedEntities() {
        return getAffectedEntities;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AnonymizationParameters that = (AnonymizationParameters) o;
        return rerunPluginPoints == that.rerunPluginPoints &&
                getAffectedEntities == that.getAffectedEntities &&
                Objects.equals(userKey, that.userKey) &&
                Objects.equals(oldUserKey, that.oldUserKey) &&
                Objects.equals(oldUserName, that.oldUserName) &&
                Objects.equals(newOwnerKey, that.newOwnerKey) &&
                Objects.equals(executor, that.executor);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userKey, rerunPluginPoints, oldUserKey, oldUserName, newOwnerKey, getAffectedEntities, executor);
    }

    @Override
    public String toString() {
        return "AnonymizationParameters{" +
                "userKey='" + userKey + '\'' +
                ", rerunPluginPoints=" + rerunPluginPoints +
                ", oldUserKey='" + oldUserKey + '\'' +
                ", oldUserName='" + oldUserName + '\'' +
                ", newOwnerKey='" + newOwnerKey + '\'' +
                ", getAffectedEntities=" + getAffectedEntities +
                ", executor=" + executor +
                '}';
    }

    public static class Builder {
        private String userKey;
        private boolean rerunPluginPoints;
        private String oldUserKey;
        private String oldUserName;
        private String newOwnerKey;
        private boolean getAffectedEntities;
        private ApplicationUser executor;

        /**
         * Key of the user that will be anonymized.
         */
        public Builder targetUser(@Nonnull String userKey) {
            this.userKey = requireNonNull(userKey);
            return this;
        }

        /**
         * Requires passing old username/user key.
         *
         * @param rerunPluginPoints indicates whether user name/key change plugin points should be re-run (when either user name/user key is already anonymized)
         */
        public Builder rerunPluginPoints(boolean rerunPluginPoints) {
            this.rerunPluginPoints = rerunPluginPoints;
            return this;
        }

        /**
         * Replace the anonymized user with this one in places
         * that always require an active user, e.g., project lead.
         */
        public Builder newOwnerKey(@Nullable String newOwnerKey) {
            this.newOwnerKey = newOwnerKey;
            return this;
        }

        /**
         * Executor of the changes. Typically, the currently logged-in user.
         */
        public Builder executor(@Nonnull ApplicationUser executor) {
            this.executor = requireNonNull(executor);
            return this;
        }

        /**
         * Old user key, used when retrying anonymization process.
         */
        public Builder oldUserKey(@Nullable String oldUserKey) {
            this.oldUserKey = oldUserKey;
            return this;
        }

        /**
         * Old user name, used when retrying anonymization process.
         */
        public Builder oldUserName(@Nullable String oldUserName) {
            this.oldUserName = oldUserName;
            return this;
        }

        /**
         * Indicates whether affected entities should be used during validation phase.
         */
        public Builder getAffectedEntities(final boolean getAffectedEntities) {
            this.getAffectedEntities = getAffectedEntities;
            return this;
        }

        public AnonymizationParameters build() {
            return new AnonymizationParameters(userKey, rerunPluginPoints, oldUserKey, oldUserName, newOwnerKey, getAffectedEntities, executor);
        }
    }
}
