package com.atlassian.jira.mail;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.Issue;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.mail.Message;
import java.util.List;

@PublicApi
public interface MailThreadManager {
    /**
     * Indicates an action in response to an incoming email.
     *
     * @since v5.2.3
     */
    enum MailAction {
        ISSUE_CREATED_FROM_EMAIL, ISSUE_COMMENTED_FROM_EMAIL
    }

    /**
     * Remembers that given messageId has been used to either create or comment on an Issue.
     * <p>
     * This creates an entry in "NotificationInstance" table that will only be removed in two cases
     * <ul>
     * <li>Issue being deleted
     * <li>Entry is deleted manually with a call to {@link #removeAssociatedEntries(Long)}
     * </ul>
     *
     * @param messageId     The incoming Message-ID
     * @param senderAddress The sender
     * @param issue         the issue that was affected (created or commented)
     * @param action        Issue created or Issue commented
     * @since v5.2.3
     */
    void storeIncomingMessageId(String messageId, String senderAddress, Issue issue, MailAction action);

    /**
     * Thread the given email which is related to the given issue.
     *
     * @param email the email to be sent
     * @param issue the issue that the email is about
     */
    void threadNotificationEmail(Email email, Issue issue);

    /**
     * Looks for an issue associated with the given message by inspecting the "In-Reply-To" header of the message.
     * <p>
     * Notifications sent from JIRA have a special form that allows us to parse out the Issue ID.
     * We also remember the incoming Message-IDs so we can tell if another recipient replies to that message,
     * but we assume only one issue can be associated with an incoming message ID.
     *
     * @param message message to analyse
     * @return associated issue or null if no issue is associated with this message.
     */
    @Nullable
    Issue getAssociatedIssueObject(Message message);

    /**
     * Looks for issues associated with the given message by inspecting the "In-Reply-To" header of the message.
     * <p>
     * Notifications sent from JIRA have a special form that allows us to parse out the Issue ID,
     * there should only be a single associated message in this case.
     * We also remember the incoming Message-IDs so we can tell if another recipient replies to that message.
     *
     * @param message message to analyse
     * @return  Potentially empty list of issues associated with this message.
     * @since v7.4.0
     */
    @Nonnull
    List<Issue> getAssociatedIssueObjects(Message message);

    /**
     * Looks for an issue associated with the given message by inspecting the "Message-ID" header of the message.
     * Assumes only one incoming email channel and therefore only the first issue found is relevant.
     *
     * @param messageId Message-ID to be checked
     * @return Issue that is already associated with this Message-ID or null if none
     */
    @Nullable
    Issue findIssueFromMessageId(String messageId);

    /**
     * Looks for issues associated with the given message by inspecting the "Message-ID" header of the message.
     *
     * @param messageId Message-ID to be checked
     * @return Potentially empty list of issues that are already associated with this Message-ID
     * @since v7.4.0
     */
    @Nonnull
    List<Issue> findIssuesFromMessageId(final String messageId);

    /**
     * Removes rows from NotificationInstance table associated with the given issue.
     * Used when we delete an issue.
     *
     * @param issueId the issue
     * @return row count
     */
    int removeAssociatedEntries(Long issueId);
}
