package com.atlassian.jira.jql.validator;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.event.issue.JqlSearchVersionEvent;
import com.atlassian.jira.jql.operand.JqlOperandResolver;
import com.atlassian.jira.jql.operator.OperatorClasses;
import com.atlassian.jira.jql.query.QueryCreationContext;
import com.atlassian.jira.jql.resolver.VersionIndexInfoResolver;
import com.atlassian.jira.jql.resolver.VersionResolver;
import com.atlassian.jira.project.version.VersionManager;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.jira.util.MessageSet;
import com.atlassian.query.clause.TerminalClause;
import com.atlassian.query.operator.Operator;
import com.google.common.annotations.VisibleForTesting;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

/**
 * Abstract version clause validator that pretty much does all the work for version validation.
 *
 * @since v4.0
 */
abstract class AbstractVersionValidator implements ClauseValidator {
    static final Set<Operator> LIKE_OPERATORS = Collections.unmodifiableSet(
            EnumSet.of(Operator.LIKE, Operator.NOT_LIKE));

    private final ValuesExistValidator versionValuesExistValidator;
    private final ValuesExistValidator versionValuesExistForLikeOperatorsValidator;
    private final SupportedOperatorsValidator supportedOperatorsValidator;
    private final WildcardCountValidator wildcardCountValidator;

    AbstractVersionValidator(final VersionResolver versionResolver, final JqlOperandResolver operandResolver,
                             final PermissionManager permissionManager, final VersionManager versionManager, final I18nHelper.BeanFactory beanFactory) {
        this.supportedOperatorsValidator = getSupportedOperatorsValidator();
        this.wildcardCountValidator = getWildcardsCountValidator(beanFactory);
        this.versionValuesExistValidator = getVersionValuesExistValidator(versionResolver, operandResolver, permissionManager, versionManager, beanFactory);
        this.versionValuesExistForLikeOperatorsValidator = getVersionValuesExistForLikeOperatorsValidator(
                versionResolver, operandResolver, permissionManager, versionManager, beanFactory);
    }

    @Nonnull
    public MessageSet validate(final ApplicationUser searcher, @Nonnull final TerminalClause terminalClause) {
        final MessageSet supportedOperatorsErrors = supportedOperatorsValidator.validate(searcher, terminalClause);
        if (supportedOperatorsErrors.hasAnyErrors()) {
            return supportedOperatorsErrors;
        }

        final MessageSet wildcardsCountErrors = wildcardCountValidator.validate(searcher, terminalClause);
        if (wildcardsCountErrors.hasAnyErrors()) {
            return wildcardsCountErrors;
        }

        ValuesExistValidator valuesExistValidator =
                LIKE_OPERATORS.contains(terminalClause.getOperator()) ? versionValuesExistForLikeOperatorsValidator : versionValuesExistValidator;
        return valuesExistValidator.validate(searcher, terminalClause);
    }

    @Nonnull
    public MessageSet validate(final TerminalClause terminalClause, final QueryCreationContext queryCreationContext) {
        final MessageSet supportedOperatorsErrors = supportedOperatorsValidator.validate(queryCreationContext.getApplicationUser(), terminalClause);
        if (supportedOperatorsErrors.hasAnyErrors()) {
            return supportedOperatorsErrors;
        }

        final MessageSet wildcardsCountErrors = wildcardCountValidator.validate(queryCreationContext.getApplicationUser(), terminalClause);
        if (wildcardsCountErrors.hasAnyErrors()) {
            return wildcardsCountErrors;
        }

        ValuesExistValidator valuesExistValidator =
                LIKE_OPERATORS.contains(terminalClause.getOperator()) ? versionValuesExistForLikeOperatorsValidator : versionValuesExistValidator;
        String operatorName = terminalClause.getOperator() != null ? terminalClause.getOperator().name() : "N/A";
        getEventPublisher().publish(new JqlSearchVersionEvent(terminalClause.getName(), operatorName));
        return valuesExistValidator.validate(terminalClause, queryCreationContext);
    }

    protected EventPublisher getEventPublisher() {
        return ComponentAccessor.getComponent(EventPublisher.class);
    }

    @VisibleForTesting
    SupportedOperatorsValidator getSupportedOperatorsValidator() {
        return new SupportedOperatorsValidator(OperatorClasses.EQUALITY_OPERATORS_WITH_EMPTY, OperatorClasses.RELATIONAL_ONLY_OPERATORS, OperatorClasses.TEXT_OPERATORS);
    }

    @VisibleForTesting
    WildcardCountValidator getWildcardsCountValidator(final I18nHelper.BeanFactory beanFactory) {
        return new WildcardCountValidator(beanFactory);
    }

    @VisibleForTesting
    VersionValuesExistValidator getVersionValuesExistValidator(final VersionResolver versionResolver,
                                                               final JqlOperandResolver operandResolver, final PermissionManager permissionManager,
                                                               final VersionManager versionManager, final I18nHelper.BeanFactory beanFactory) {
        return new VersionValuesExistValidator(operandResolver, new VersionIndexInfoResolver(versionResolver),
                permissionManager, versionManager, beanFactory);
    }

    @VisibleForTesting
    VersionValuesExistValidator getVersionValuesExistForLikeOperatorsValidator(final VersionResolver versionResolver,
                                                               final JqlOperandResolver operandResolver, final PermissionManager permissionManager,
                                                               final VersionManager versionManager, final I18nHelper.BeanFactory beanFactory) {
        return new VersionValuesExistValidator(operandResolver, new VersionIndexInfoResolver(versionResolver, versionResolver::getIdsFromNameWildcard),
                permissionManager, versionManager, beanFactory);
    }
}
