package com.atlassian.jira.jql.query;

import com.atlassian.jira.jql.operand.QueryLiteral;
import com.atlassian.jira.jql.operator.OperatorClasses;
import com.atlassian.jira.jql.util.IndexValueConverter;
import com.atlassian.query.operator.Operator;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

/**
 * Creates relational queries for clauses whose values are exactly the same as the indexed value (e.g. votes and durations).
 *
 * @since v4.0
 */
public class ActualValueRelationalQueryFactory extends AbstractActualValueOperatorQueryFactory implements OperatorSpecificQueryFactory {
    private static final Logger log = LoggerFactory.getLogger(ActualValueRelationalQueryFactory.class);
    private static final RangeQueryFactory<String> RANGE_QUERY_FACTORY = RangeQueryFactory.stringRangeQueryFactory();

    private final String emptyIndexValue;

    public ActualValueRelationalQueryFactory(final IndexValueConverter indexValueConverter) {
        this(indexValueConverter, null);
    }

    /**
     * @param indexValueConverter the value converter
     * @param emptyIndexValue     specify this if the index field uses a special empty value that we must exclude from the
     *                            range queries generated by relational operators.
     */
    public ActualValueRelationalQueryFactory(final IndexValueConverter indexValueConverter, final String emptyIndexValue) {
        super(indexValueConverter);
        this.emptyIndexValue = emptyIndexValue;
    }

    public QueryFactoryResult createQueryForSingleValue(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        if (!handlesOperator(operator)) {
            log.debug(String.format("ActualValueRelationalQueryFactory does not support operator '%s'.", operator.getDisplayString()));
            return QueryFactoryResult.createFalseResult();
        }

        final List<String> values = getIndexValues(rawValues);

        // if no index values were contained in the literals, the resultant list will be empty
        if (values.isEmpty()) {
            return QueryFactoryResult.createFalseResult();
        }

        // most operators only expect one value
        final String value = values.get(0);

        // if we somehow got null as the value, we don't support empty so just return an empty query
        if (value == null) {
            log.debug("Relation operators do not support empty values");
            return QueryFactoryResult.createFalseResult();
        }

        Query query = RANGE_QUERY_FACTORY.get(operator, fieldName, value);

        // exclude the emptyIndexValue (if specified) from the range
        if (emptyIndexValue != null) {
            final BooleanQuery.Builder combined = new BooleanQuery.Builder();
            combined.add(query, BooleanClause.Occur.MUST);
            combined.add(new TermQuery(new Term(fieldName, emptyIndexValue)), BooleanClause.Occur.MUST_NOT);
            query = combined.build();
        }
        return new QueryFactoryResult(query);
    }

    public QueryFactoryResult createQueryForMultipleValues(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        log.debug("Relation operators do not support multivalue operands");
        return QueryFactoryResult.createFalseResult();
    }

    public QueryFactoryResult createQueryForEmptyOperand(final String fieldName, final Operator operator) {
        log.debug("Relation operators do not support multivalue empty");
        return QueryFactoryResult.createFalseResult();
    }

    public boolean handlesOperator(final Operator operator) {
        return OperatorClasses.RELATIONAL_ONLY_OPERATORS.contains(operator);
    }
}
