package com.atlassian.jira.issue.search;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.web.bean.PagerFilter;

import java.util.List;

@PublicApi
public class IssueSearchResultsFactory {

    /**
     * Construct searchResults using a list of issues.  The issues returned by {@link #getIssues()} will
     * be a subset of the issues passed in.
     *
     * @param issues      A list of {@link com.atlassian.jira.issue.Issue} objects
     * @param pagerFilter Representing which issues to limit the results to
     */
    public static SearchResults<Issue> create(final List<Issue> issues, final PagerFilter<Issue> pagerFilter) {
        // Reset the pager filters start value if the current value is not sane
        if (issues.size() < pagerFilter.getStart()) {
            pagerFilter.setStart(0);
        }
        return new SearchResults(pagerFilter.getCurrentPage(issues), issues.size(), pagerFilter.getMax(), pagerFilter.getStart());
    }

    /**
     * Construct searchResults using the issues that should be displayed, and the 'total' number of issues.
     * This is used when a search does not wish to load the entire list of issues into memory.
     *
     * @param issuesInPage    A list of {@link com.atlassian.jira.issue.Issue} objects
     * @param totalIssueCount The count of the number of issues returned
     * @param pagerFilter     Representing the users preference for paging
     */
    public static SearchResults<Issue> create(final List<Issue> issuesInPage, final int totalIssueCount, final PagerFilter pagerFilter) {
        // Reset the pager filters start value if the current value is not sane
        if (totalIssueCount < pagerFilter.getStart()) {
            pagerFilter.setStart(0);
        }
        return new SearchResults(issuesInPage, totalIssueCount, pagerFilter.getMax(), pagerFilter.getStart());
    }

    /**
     * Construct searchResults using the issues that should be displayed, and the 'total' number of issues.
     * This is used when we do a stable search and want to return a max of the selected page's length, not the
     * stable search limit.
     *
     * @param issuesInPage    A list of {@link com.atlassian.jira.issue.Issue} objects
     * @param totalIssueCount The count of the number of issues returned
     * @param maxIssueCount   The maximum number of issues to include in the search
     * @param startIndex      The index of the first issue in the search
     */
    public static SearchResults<Issue> create(final List<Issue> issuesInPage, final int totalIssueCount, final int maxIssueCount, int startIndex) {
        // Reset the pager filters start value if the current value is not sane
        if (totalIssueCount < startIndex) {
            startIndex = 0;
        }
        return new SearchResults(issuesInPage, totalIssueCount, maxIssueCount, startIndex);
    }

    private IssueSearchResultsFactory() {
    }
}
