package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.converters.DoubleConverter;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.DocumentConstants;
import com.atlassian.jira.lucenelegacy.NumericUtils;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.DoubleDocValuesField;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.SortedDocValuesField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.util.BytesRef;

import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * A simple custom field indexer for the number custom fields
 *
 * @since v4.0
 */
public class NumberCustomFieldIndexer extends AbstractCustomFieldIndexer {

    private final DoubleConverter doubleConverter;
    private final boolean skipIndexingNull;

    public NumberCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager,
                                    final CustomField customField,
                                    final DoubleConverter doubleConverter,
                                    final boolean skipIndexingNull) {
        super(fieldVisibilityManager, notNull("customField", customField));
        this.doubleConverter = notNull("doubleConverter", doubleConverter);
        this.skipIndexingNull = skipIndexingNull;
    }

    public NumberCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager,
                                    final CustomField customField,
                                    final DoubleConverter doubleConverter) {
        this(fieldVisibilityManager, customField, doubleConverter, false);
    }

    public void addDocumentFieldsSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, true);
    }

    public void addDocumentFieldsNotSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, false);
    }

    @Override
    public Boolean skipsIndexingNull() {
        return skipIndexingNull;
    }

    private void addDocumentFields(final Document doc, final Issue issue, final boolean searchable) {
        Object value = customField.getValue(issue);
        if (value != null) {
            final String string = doubleConverter.getStringForLucene((Double) value);
            if (searchable) {
                doc.add(new StringField(getDocumentFieldId(), string, Field.Store.YES));
                doc.add(new DoubleDocValuesField(getDocumentFieldId(), (Double) value));
            } else {
                doc.add(new StoredField(getDocumentFieldId(), string));
            }
            if (searchable) {
                // We store this field twice in the index, because historically JIRA indexed the number in a format that did not sort negative numbers correctly.
                // We now index the field using the properly encoded format to allow sorting. To keep backward compatibility with third party plugins we retain
                // the old stored value and also index the old value, although these plugins should migrate to using NumberCustomFieldClauseQueryFactory.
                String encoded = NumericUtils.doubleToPrefixCoded((Double) value);
                doc.add(new StringField(DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId(), encoded, Field.Store.NO));
                doc.add(new SortedDocValuesField(DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId(), new BytesRef(encoded)));
            }
        } else {
            if (searchable && !skipIndexingNull) {
                String encoded = NumericUtils.doubleToPrefixCoded(Double.MAX_VALUE);
                doc.add(new StringField(DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId(), encoded, Field.Store.NO));
                doc.add(new SortedDocValuesField(DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId(), new BytesRef(encoded)));
            }
        }
    }
}