package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.annotations.PublicSpi;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.vdi.CustomFieldPrefetchedData;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.indexers.FieldIndexer;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;

/**
 * A base-class for custom field indexers that performs the logic of checking if the CustomField isVisibleAndInScope
 * and then delegates off to the base class to add the correct document fields.
 *
 * @since v4.0
 */
@PublicSpi
public abstract class AbstractCustomFieldIndexer implements FieldIndexer {
    private final FieldVisibilityManager fieldVisibilityManager;
    protected final CustomField customField;

    protected AbstractCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager, final CustomField customField) {
        this.fieldVisibilityManager = fieldVisibilityManager;
        this.customField = customField;
    }

    public String getId() {
        return customField.getId();
    }

    public String getDocumentFieldId() {
        return customField.getId();
    }

    /**
     *
     * @deprecated since 8.10 implement {@link #addDocumentFieldsSearchable(Document, Issue, CustomFieldPrefetchedData)}  instead
     */
    @Deprecated
    public void addDocumentFieldsSearchable(final Document doc, final Issue issue) {
        throw new UnsupportedOperationException();
    }

    /**
     *
     * @deprecated since 8.10 implement {@link #addDocumentFieldsNotSearchable(Document, Issue, CustomFieldPrefetchedData)}  instead
     */
    @Deprecated
    public void addDocumentFieldsNotSearchable(final Document doc, final Issue issue) {
        throw new UnsupportedOperationException();
    }

    /**
     *
     * @deprecated since 8.12 implement {@link #addDocumentFieldsNotSearchable(Document, Issue, CustomFieldPrefetchedData)}  instead
     */
    @Deprecated
    protected void addDocumentFieldsSearchable(final Document doc, final Issue issue, final com.atlassian.jira.issue.customfields.CustomFieldPrefetchedData prefetchedData) {
        addDocumentFieldsSearchable(doc, issue);
    }

    /**
     *
     * @deprecated since 8.12 implement {@link #addDocumentFieldsNotSearchable(Document, Issue, CustomFieldPrefetchedData)}  instead
     */
    @Deprecated
    protected void addDocumentFieldsNotSearchable(final Document doc, final Issue issue, final com.atlassian.jira.issue.customfields.CustomFieldPrefetchedData prefetchedData) {
        addDocumentFieldsNotSearchable(doc, issue);
    }

    protected void addDocumentFieldsSearchable(final Document doc, final Issue issue, final CustomFieldPrefetchedData prefetchedData) {
        addDocumentFieldsSearchable(doc, issue);
    }

    protected void addDocumentFieldsNotSearchable(final Document doc, final Issue issue, final CustomFieldPrefetchedData prefetchedData) {
        addDocumentFieldsNotSearchable(doc, issue);
    }

    @Override
    public final void addIndex(final Document doc, final Issue issue) {
        if (isFieldVisibleAndInScope(issue)) {
            addDocumentFieldsSearchable(doc, issue);
        } else {
            addDocumentFieldsNotSearchable(doc, issue);
        }
    }

    @Override
    public void addIndex(final Document doc, final Issue issue, final com.atlassian.jira.issue.customfields.CustomFieldPrefetchedData prefetchedData) {
        if (isFieldVisibleAndInScope(issue)) {
            addDocumentFieldsSearchable(doc, issue, prefetchedData);
        } else {
            addDocumentFieldsNotSearchable(doc, issue, prefetchedData);
        }
    }

    @Override
    public void addIndex(final Document doc, final Issue issue, final CustomFieldPrefetchedData prefetchedData) {
        if (isFieldVisibleAndInScope(issue)) {
            addDocumentFieldsSearchable(doc, issue, prefetchedData);
        } else {
            addDocumentFieldsNotSearchable(doc, issue, prefetchedData);
        }
    }

    public boolean isFieldVisibleAndInScope(final Issue issue) {
        // Check Field Visibility first as it is considered more likely to return false.
        return fieldVisibilityManager.isFieldVisible(getId(), issue) && isRelevantForIssue(issue);
    }

    protected boolean isRelevantForIssue(final Issue issue) {
        return customField.isRelevantForIssueContext(issue);
    }
}
