package com.atlassian.jira.issue.fields;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.admin.RenderableProperty;
import com.atlassian.jira.exception.DataAccessException;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.context.IssueContext;
import com.atlassian.jira.issue.context.JiraContextNode;
import com.atlassian.jira.issue.customfields.CustomFieldSearcher;
import com.atlassian.jira.issue.customfields.CustomFieldType;
import com.atlassian.jira.issue.customfields.option.Options;
import com.atlassian.jira.issue.customfields.view.CustomFieldParams;
import com.atlassian.jira.issue.fields.config.FieldConfig;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.issue.fields.renderer.RenderableField;
import com.atlassian.jira.issue.fields.rest.RestAwareField;
import com.atlassian.jira.issue.fields.rest.RestFieldOperations;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.search.ClauseNames;
import com.atlassian.jira.issue.search.SearchContext;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.util.ErrorCollection;
import com.opensymphony.module.propertyset.PropertySet;
import org.ofbiz.core.entity.GenericValue;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.sql.Timestamp;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Custom Field interface.
 * <p>
 * Typically one obtains a CustomField using {@link com.atlassian.jira.issue.CustomFieldManager},
 * eg. {@link com.atlassian.jira.issue.CustomFieldManager#getCustomFieldObjectByName(String)}
 * <p>
 * To create or update an instance of a CustomField for an issue use {@link #createValue(com.atlassian.jira.issue.Issue, Object)}
 * or {@link OrderableField#updateValue(com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem, com.atlassian.jira.issue.Issue, com.atlassian.jira.issue.ModifiedValue, com.atlassian.jira.issue.util.IssueChangeHolder)}.
 *
 * @see CustomFieldType CustomFieldType - The type of custom field (text, number, user picker etc).
 */
@PublicApi
public interface CustomField extends NavigableField, HideableField, DefaultValueField<Object>, RenderableField, RequirableField, OrderableField<Object>, RestAwareField, RestFieldOperations {
    // ------------------------------------------------------------------------------------------------------- Constants
    String ENTITY_CF_TYPE_KEY = "customfieldtypekey";
    String ENTITY_CUSTOM_FIELD_SEARCHER = "customfieldsearcherkey";
    String ENTITY_NAME = "name";
    String ENTITY_ISSUETYPE = "issuetype";
    String ENTITY_PROJECT = "project";
    String ENTITY_ID = "id";
    String ENTITY_DESCRIPTION = "description";
    String ENTITY_TABLE_NAME = "CustomField";
    String ENTITY_LAST_VALUE_UPDATE = "lastvalueupdate";
    String ENTITY_ISSUES_WITH_VALUE = "issueswithvalue";

    /**
     * Determines if this custom field is within the scope of the given project, and list of Issue Types.
     *
     * @param project      The project.
     * @param issueTypeIds A list of IssueType ids.
     * @return {@code true} if this custom field is within the given scope.
     */
    boolean isInScope(Project project, List<String> issueTypeIds);

    /**
     * Determines if this custom field is within the scope of the given project, and Issue Type.
     *
     * @param projectId   The project.
     * @param issueTypeId The issueTypeId
     * @return {@code true} if this custom field is within the given scope.
     * @see #isRelevantForIssueContext(com.atlassian.jira.issue.context.IssueContext)
     */
    boolean isInScope(long projectId, String issueTypeId);

    /**
     * Determines if this custom field is within the scope of the given project, and list of Issue Types.
     * <p>
     * If the project is null, then it is treated as <tt>any project</tt>.
     * If the issueTypeIds list is null or an empty list, then it is treated as <tt>any issue type</tt>.
     * </p>
     * <p>
     * If the passed project is <tt>any project</tt>, this method will search in all the {@link FieldConfigScheme}
     * of this custom field, ignoring the projects that they apply to (since the given project is <tt>any</tt>)
     * and looking for at least one of them that applies to at least one of the given issue type ids.
     * </p>
     * <p>
     * If the passed list of issue types is <tt>any issue type</tt>, this method will search for at least one {@link FieldConfigScheme}
     * that applies to the given project, ignoring the issue types that it applies to (since the given issue type ids are <tt>any</tt>).
     * </p>
     * <p>
     * If both the project and issue types are <tt>any</tt>, the question being asked is "is this custom field
     * in the scope of any project and any issue type?", which will always be true.
     * </p>
     */
    boolean isInScopeForSearch(@Nullable Project project, @Nullable List<String> issueTypeIds);

    /**
     * Determines whether this custom field is in scope.
     *
     * @param searchContext search context
     * @return true if this field is in scope
     */
    boolean isInScope(SearchContext searchContext);

    /**
     * Returns a generic value that represents this custom field
     *
     * @return generic value of this custom field
     * @deprecated Use {@link #getName()}, {@link #getDescription()}, etc. Since v3.0.
     */
    @Deprecated
    GenericValue getGenericValue();

    /**
     * This method compares the values of this custom field in two given issues.
     * <p>
     * Returns a negative integer, zero, or a positive integer as the value of first issue is less than, equal to,
     * or greater than the value of the second issue.
     * <p>
     * If either of given issues is null a IllegalArgumentException is thrown.
     *
     * @param issue1 issue to compare
     * @param issue2 issue to compare
     * @return a negative integer, zero, or a positive integer as the value of first issue is less than, equal to, or
     * greater than the value of the second issue
     * @throws IllegalArgumentException if any of given issues is null
     */
    public int compare(Issue issue1, Issue issue2) throws IllegalArgumentException;

    //------------------------------------------------- methods used for create / edit, which use Collections of Strings

    /**
     * Get the custom field string values that are relevant to this particular custom field
     *
     * @param customFieldValuesHolder containing all params
     * @return a {@link CustomFieldParams} of {@link String} objects
     */
    public CustomFieldParams getCustomFieldValues(Map customFieldValuesHolder);

    /**
     * Retrieves and returns the Object representing the this CustomField value for the given issue.
     * See {@link CustomFieldType#getValueFromIssue(CustomField, Issue)}.
     * This is only used to communicate with the 'view' JSP. Multiselects will return a list, dates a date, etc.
     *
     * @param issue issue to retrieve the value from
     * @return Object representing the this CustomField value for the given issue
     * @see #getValueFromParams(java.util.Map)
     */
    public Object getValue(Issue issue);

    /**
     * Removes this custom field and returns a set of issue IDs of all issues that are affected by removal of this
     * custom field.
     *
     * @return a set of issue IDs of affected issues
     * @throws DataAccessException if removal of generic value fails
     * @deprecated Use {@link com.atlassian.jira.issue.CustomFieldManager#removeCustomField(CustomField)} instead. Since v7.0.
     */
    @Deprecated
    public Set<Long> remove();

    /**
     * Returns options for this custom field if it is
     * of {@link com.atlassian.jira.issue.customfields.MultipleCustomFieldType} type. Otherwise returns null.
     * <p>
     * As this is just used by the view layer, it can be a list of objects
     *
     * @param key             not used
     * @param jiraContextNode JIRA context node
     * @return options for this custom field if it is of {@link com.atlassian.jira.issue.customfields.MultipleCustomFieldType} type, null otherwise
     */
    public Options getOptions(String key, JiraContextNode jiraContextNode);

    /**
     * Returns the 1i8n'ed description of this custom field. To render views for the custom field description, prefer
     * {@link #getDescriptionProperty()}.
     *
     * @return the description of this custom field
     */
    String getDescription();

    /**
     * Returns the description of this custom field by reading {@link #ENTITY_DESCRIPTION} of the underlying generic value.
     *
     * @return the description of this custom field
     */
    String getUntranslatedDescription();

    /**
     * Returns the title of this custom field.
     *
     * @return the title of this custom field
     */
    String getFieldName();

    /**
     * Returns the name of this custom field by reading {@link #ENTITY_NAME} of the underlying generic value.
     *
     * @return the name of this custom field
     */
    String getUntranslatedName();

    /**
     * Returns a {@code RenderableProperty} for rendering this custom field's description.
     *
     * @return a read-only RenderableProperty
     * @since v5.0.7
     */
    @Nonnull
    RenderableProperty getDescriptionProperty();

    /**
     * Returns a {@code RenderableProperty} for rendering this custom field's untranslated description, for admin.
     *
     * @return a read-only RenderableProperty
     * @since v5.0.7
     */
    @Nonnull
    RenderableProperty getUntranslatedDescriptionProperty();

    /**
     * Retrieves the {@link CustomFieldSearcher} for this custom field.
     * The searcher, if found is initialized with this custom field before it is returned.
     *
     * @return found custom field searcher or null, if none found
     */
    CustomFieldSearcher getCustomFieldSearcher();

    /**
     * Returns true if this custom field can be edited, false otherwise.
     *
     * @return true if this custom field can be edited, false otherwise
     */
    boolean isEditable();

    /**
     * Returns ID of this custom field.
     *
     * @return ID of this custom field
     */
    Long getIdAsLong();

    /**
     * Returns the latest date of either:
     * - creation date of an issue with a non default value of the custom field
     * - issue update date when value of the custom field has been changed
     *
     * @return last update of the custom field
     */
    default Timestamp getLastValueUpdate() {
        return null;
    }

    /**
     * Returns number of issues containing at least one non empty value for the custom field.
     *
     * @return number of issues with the non empty custom field
     */
    default Long getIssuesWithValue() {
        return null;
    }

    /**
     * Returns a list of configuration schemes.
     *
     * @return a list of {@link FieldConfigScheme} objects.
     */
    @Nonnull
    List<FieldConfigScheme> getConfigurationSchemes();

    /**
     * Returns options for this custom field if it is
     * of {@link com.atlassian.jira.issue.customfields.MultipleCustomFieldType} type. Otherwise returns null.
     * <p>
     * As this is just used by the view layer, it can be a list of objects
     *
     * @param key         not used
     * @param config      relevant field config
     * @param contextNode JIRA context node
     * @return options for this custom field if it is of {@link com.atlassian.jira.issue.customfields.MultipleCustomFieldType} type, null otherwise
     */
    Options getOptions(String key, FieldConfig config, JiraContextNode contextNode);

    /**
     * Returns a relevant {@link FieldConfig} for the given issue. If the field <strong>has</strong>
     * a config for the issue then one will be returned, otherwise null is returned.
     * <p>
     * For example, if we have 2 projects: project A and project B, and a custom field is configured to be only
     * applicable to project A, calling getRelevantConfig with an issue from project A should return the config
     * (i.e. <strong>not</strong> null). Calling this method with an issue from project B <strong>should</strong>
     * return null.
     *
     * @param issue issue whose project and issue type will be used to check if the field has a config
     * @return an instance of {@link FieldConfig} representing the configuration of the field for issue's
     * project/issue type. If the field does not have a config for issue's project/issue type, null is returned.
     * @see #isRelevantForIssueContext(com.atlassian.jira.issue.context.IssueContext)
     */
    FieldConfig getRelevantConfig(Issue issue);

    /**
     * Validates relevant parameters on custom field type of this custom field. Any errors found are added to the given
     * errorCollection.
     * See {@link CustomFieldType#validateFromParams(CustomFieldParams, ErrorCollection, FieldConfig)}
     *
     * @param actionParameters action parameters
     * @param errorCollection  error collection to add errors to
     * @param config           field config
     */
    void validateFromActionParams(Map actionParameters, ErrorCollection errorCollection, FieldConfig config);

    /**
     * Returns a list of {@link Project}s directly associated with this {@link CustomField}.
     *
     * <p>
     * This method can return an empty result when the {@link CustomField} is associated with all {@link Project}s.
     * Please use {@link #isAllProjects()} prior to this method to detect this."
     * </p>
     *
     * @return a list of {@link Project}s directly associated with this {@link CustomField}.
     */
    @Nonnull
    List<Project> getAssociatedProjectObjects();

    /**
     * Returns a list of {@link IssueType}s directly associated with this {@link CustomField}.
     *
     * <p>
     * In case when this {@link CustomField} {@link #isAllIssueTypes()} it will be associated
     * with all {@link IssueType}s but this method can still return empty result.
     * </p>
     *
     * @return a list of {@link IssueType}s directly associated with this {@link CustomField}.
     */
    @Nonnull
    List<IssueType> getAssociatedIssueTypes();

    /**
     * @see #getAssociatedIssueTypes()
     * @deprecated As of 7.2, please use {@link #getAssociatedIssueTypes()}
     */
    @Deprecated
    List<IssueType> getAssociatedIssueTypeObjects();

    /**
     * Returns true if this custom field applies for all projects and all issue types.
     *
     * @return true if it is in all projects and all issue types, false otherwise.
     */
    boolean isGlobal();

    /**
     * Checks whether this custom field applies for all projects. It returns true if it applies for all projects
     * for any field configuration scheme, false otherwise.
     *
     * @return true if it applies for all projects for any field configuration scheme, false otherwise.
     */
    boolean isAllProjects();

    /**
     * Returns true if it applies for all issue types, false otherwise.
     *
     * @return true if it applies for all issue types, false otherwise.
     */
    boolean isAllIssueTypes();

    /**
     * Returns true if any configuration scheme returned by {@link #getConfigurationSchemes()} is enabled.
     *
     * @return true if any configuration scheme is enabled, false otherwise
     */
    boolean isEnabled();

    /**
     * Looks up the {@link com.atlassian.jira.issue.customfields.CustomFieldType}. It can return null if the custom
     * field type cannot be found in the {@link com.atlassian.jira.issue.CustomFieldManager}.
     *
     * @return custom field type
     */
    CustomFieldType getCustomFieldType();

    /**
     * Returns true if the custom field has a config for the Project and Issue Type of the given IssueContext.
     * <p>
     * This is equivalent to calling
     * <pre>  getRelevantConfig(issueContext) != null</pre>
     * but in general can run faster because it does not have to resolve the actual FieldConfig.
     *
     * @param issueContext IssueContext whose project and issue type will be used to check if the field has a config
     * @return an instance of {@link FieldConfig} representing the configuration of the field for issue's
     * project/issue type. If the field does not have a config for issue's project/issue type, null is returned.
     * @see #getRelevantConfig(com.atlassian.jira.issue.context.IssueContext)
     * @see #isInScope(long, String)
     */
    boolean isRelevantForIssueContext(IssueContext issueContext);

    /**
     * Returns the relevant field config of this custom field for the give issue context
     *
     * @param issueContext issue context to find the relevant field config for
     * @return the relevant field config of this custom field for the give issue context
     * @see #isRelevantForIssueContext(com.atlassian.jira.issue.context.IssueContext)
     */
    FieldConfig getRelevantConfig(IssueContext issueContext);

    /**
     * The {@link FieldConfig} that is relevent to <strong>all</strong> issue contexts in the search context
     * Checks that all configs within search context are the same - i.e. all null or all the same config.
     * <p>
     * Returns null if any two configs are different.
     * <p>
     * Note: null config is not equal to non-null config. Previously, a non-null config was returned even if the first
     * config(s) was null.
     *
     * @param searchContext search context
     * @return null if any two configs are different
     */
    FieldConfig getReleventConfig(SearchContext searchContext);

    /**
     * Return the JQL clause names that this custom field should be recognized by.
     *
     * @return the clause names this custom field should be recognized by.
     */
    ClauseNames getClauseNames();

    // Get the property set associated with this custom field
    PropertySet getPropertySet();
}
