package com.atlassian.jira.issue;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;
import io.atlassian.fugue.Either;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Provides validation methods for attachments.
 *
 * @since v6.4
 */
@PublicApi
@ParametersAreNonnullByDefault
public interface AttachmentValidator {

    /**
     * Verifies if user may create attachment for given issue.
     *
     * @param user            user who creates attachment
     * @param issue           target issue
     * @param errorCollection error collections that will be filled in case of error
     * @return true if user can create attachments
     */
    boolean canCreateAttachments(@Nullable ApplicationUser user, Issue issue, ErrorCollection errorCollection);

    /**
     * Check if the given user has permission to add an attachment during a workflow transition to a step with given id.
     * This checks if the user is able to add an attachment in either the origin status or the target status of the workflow transition.
     *
     * @param user               user who creates attachment
     * @param issue              issue before workflow transition described by actionDescriptor
     * @param nextStatusId       the status the issue will be in after workflow transition; if null the logic is the same
     *                           as in {@link AttachmentValidator#canCreateAttachments(ApplicationUser, Issue, ErrorCollection)}
     * @param errorCollection    error collections that will be filled in case of error
     * @return true if user can create attachments in origin status or the status after transition
     * @since 8.3
     */
    @ExperimentalApi
    boolean canCreateAttachments(@Nullable ApplicationUser user, @Nonnull Issue issue, @Nullable String nextStatusId, @Nonnull ErrorCollection errorCollection);

    /**
     * Verifies if user may create temporary attachment for given issue or project (in case if the issue is being
     * created).
     *
     * @param user            user who creates temporary attachment
     * @param issueOrProject  target issue or project (in case if the issue is being created)
     * @param errorCollection error collections that will be filled in case of error
     * @return true if user can create temporary attachments
     */
    boolean canCreateTemporaryAttachments(@Nullable ApplicationUser user, Either<Issue, Project> issueOrProject,
                                          ErrorCollection errorCollection);
}
