package com.atlassian.jira.database;

import com.google.common.collect.ImmutableMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;

import static org.apache.commons.lang3.math.NumberUtils.toInt;

/**
 * An enum representing the supported database vendors that JIRA can connect to.
 *
 * @since v7.0
 */
public enum DatabaseVendor {
    POSTGRES("Postgres", new PostgreSQLVersionStrategy()),
    ORACLE("Oracle", new OracleVersionStrategy()),
    SQL_SERVER("SQL Server", new MsSqlVersionStrategy()),
    H2("H2", new GenericVersionStrategy()),
    MY_SQL("MySQL", new GenericVersionStrategy()),
    UNSUPPORTED("unsupported database", null),
    FAKE_DATABASE_FOR_TESTING("fakedatabase", new GenericVersionStrategy());

    final String humanReadableName;
    final DatabaseVersionStrategy versionStrategy;

    DatabaseVendor(final String humanReadableName, DatabaseVersionStrategy versionStrategy) {
        this.humanReadableName = humanReadableName;
        this.versionStrategy = versionStrategy;
    }

    public String getHumanReadableName() {
        return humanReadableName;
    }

    /**
     * This is a public static delegate to private static method getSQLServerVersion().
     * It's there to satisfy API checker / backwards compatibility.
     *
     * @param version The version as a String, e.g.: "10.50.1600.1".
     * @return marketing version name e.g "2008 R2"
     * @deprecated since 8.3, to be removed in 9.0
     */
    @Deprecated
    public static String getSQLServerVersionIfExists(final String version) {
        try {
            return SQL_SERVER.getVersion(version);
        } catch (final IllegalArgumentException e) {
            return version;
        }
    }

    /**
     * Gets a release version of a database, i.e.:
     * - minor.minor (without micro/bugfix position)
     * or
     * - marketing name for a release
     * Note: this method is overridden for some vendors
     *
     * @param version The version string as retrieved from the DatabaseAccessor
     * @return release version
     * @throws IllegalArgumentException in case of failure, message explains the reason
     */
    public String getVersion(final String version) {
        return versionStrategy.getVersion(version);
    }

    /**
     * Gets a release version of a database like getVersion() method, but does not throw in case of failure.
     *
     * @param version The version string as retrieved from the DatabaseAccessor
     * @return release version or input version in case of failure
     */
    public String getHumanReadableVersion(final String version) {
        try {
            return getVersion(version);
        } catch (final IllegalArgumentException e) {
            return version;
        }
    }

    private interface DatabaseVersionStrategy {
        String getVersion(final String version);
    }

    private static class MsSqlVersionStrategy implements DatabaseVersionStrategy {
        private static final Map<String, String> MSSQL_VERSION_ALIASES =
                ImmutableMap.<String, String>builder()
                        .put("15.*", "2019")
                        .put("14.*", "2017")
                        .put("13.*", "2016")

                        //This is also the database version Azure pretends to be by default when I tested it  on 14-05-2018
                        //It's possible for users to set it to be a different version however
                        //https://azure.microsoft.com/en-us/blog/default-compatibility-level-140-for-azure-sql-databases/
                        .put("Azure 12.*", "Azure")
                        .put("12.*", "2014")

                        .put("11.*", "2012")
                        .put("10.5.+", "2008 R2")
                        .put("10.[0-4].*", "2008")
                        .put("9.*", "2005")
                        .put("8.*", "2000").build();

        /**
         * Microsoft SQL server markets their databases under different version names than their database reports their version
         * So this is so we can show error messages referencing the version they're familiar with (the marketing version)
         * <p>
         * Takes a Microsoft SQL Server version string and gets its corresponding marketing version name
         *
         * @param version The version as a String, e.g.: "10.50.1600.1".
         * @return marketing version name e.g "2008 R2"
         */
        public String getVersion(final String version) {
            return MSSQL_VERSION_ALIASES
                    .entrySet()
                    .stream()
                    .filter(entry -> version.matches(entry.getKey()))
                    .map(Map.Entry::getValue)
                    .findFirst()
                    .orElseThrow(() -> new IllegalArgumentException(
                            String.format("the version number retrieved from the database isn't a known version of SQL Server (version string: \"%s\")",
                                    StringEscapeUtils.escapeJava(version))));
        }
    }

    private static class PostgreSQLVersionStrategy implements DatabaseVersionStrategy {
        private static final Pattern VERSION_PATTERN = Pattern.compile("(\\d+(\\.\\d+)*).*");

        /**
         * * Gets a release version of a database without bugfix position, e.g.:
         * - major.minor (for versions lower and equal 9)
         * and
         * - major (for versions greater and equal 10)
         *
         * @param version version as a String, e.g.: "9.6.1" or "10.4".
         * @return release version without bugfix position
         * @throws IllegalArgumentException in case of failure
         */
        public String getVersion(final String version) {
            final Matcher matcher = VERSION_PATTERN.matcher(version);
            if (!matcher.matches()) {
                throw new IllegalArgumentException(
                        String.format("the version string retrieved from the database isn't a version number (version string: \"%s\")",
                                StringEscapeUtils.escapeJava(version)));
            }

            final String[] parts = StringUtils.split(matcher.group(1), ".");
            final int major = parts.length > 0 ? toInt(parts[0]) : 0;
            if (major < 10) {
                final int minor = parts.length > 1 ? toInt(parts[1]) : 0;
                return major + "." + minor;
            } else {
                return String.valueOf(major);
            }
        }
    }

    private static class OracleVersionStrategy implements DatabaseVersionStrategy {
        private static final Pattern ORACLE_MAJOR_MINOR_PATTERN = Pattern.compile("^.*?(\\d+\\.\\d+)\\.\\d+\\.\\d+.*?", Pattern.DOTALL);

        private static final Map<String, String> ORACLE_VERSION_ALIASES =
                ImmutableMap.<String, String>builder()
                        .put("19.0", "19C")
                        .put("18.0", "18C")
                        .put("12.2", "12C R2")
                        .put("12.1", "12C R1")
                        .put("11.2", "11G R2")
                        .put("11.1", "11G R1").build();

        /**
         * Oracle markets their databases under different version names than their database reports.
         * This here translates Oracle versions so we can show error messages referencing the version they're familiar with (the marketing version)
         * <p>
         * Takes a Oracle Database version string and gets its corresponding marketing version name
         *
         * @param version The version as a String, e.g.: "Oracle Database 12c Standard Edition Release 12.2.0.1.0 - 64bit Production".
         * @return marketing version name e.g "12c R1"
         */
        public String getVersion(final String version) {
            final Matcher m = ORACLE_MAJOR_MINOR_PATTERN.matcher(version);
            if (!m.matches()) {
                throw new IllegalArgumentException(
                        String.format("the version string retrieved from the database does not seem to contain a version number (version string: \"%s\")",
                                StringEscapeUtils.escapeJava(version)));
            }
            return ORACLE_VERSION_ALIASES
                    .entrySet()
                    .stream()
                    .filter(entry -> m.group(1).equals(entry.getKey()))
                    .map(Map.Entry::getValue)
                    .findFirst()
                    .orElseThrow(() -> new IllegalArgumentException(
                            String.format("the version number retrieved from the database isn't a known version of Oracle Database (version string: \"%s\")",
                                    StringEscapeUtils.escapeJava(version))));
        }
    }

    private static class GenericVersionStrategy implements DatabaseVersionStrategy {
        private static final Pattern VERSION_PATTERN = Pattern.compile("(\\d+(\\.\\d+)*).*");

        @Override
        public String getVersion(String version) {
            final Matcher matcher = VERSION_PATTERN.matcher(version);
            if (matcher.matches()) {
                String[] parts = StringUtils.split(matcher.group(1), ".");
                final int major = parts.length > 0 ? toInt(parts[0]) : 0;
                final int minor = parts.length > 1 ? toInt(parts[1]) : 0;
                return major + "." + minor;
            }
            throw new IllegalArgumentException(
                    String.format("the version number retrieved from the database isn't a version number we expected to get (major.minor.micro) (version string: \"%s\")",
                            StringEscapeUtils.escapeJava(version)));
        }
    }
}