package com.atlassian.jira.config.util;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.util.PathUtils;

import java.io.File;

/**
 * Responsible for determining the current location of JIRA indexes.
 */
@PublicApi
public interface IndexPathManager {
    public static final String INDEXES_DIR = PathUtils.joinPaths(JiraHome.CACHES, "indexesV1");

    static final class Directory {
        public static final String ISSUES_SUBDIR = "issues";
        public static final String COMMENTS_SUBDIR = "comments";
        public static final String PLUGINS_SUBDIR = "plugins";
        public static final String ENTITIES_SUBDIR = "entities";
        public static final String CHANGE_HISTORY_SUBDIR = "changes";
        public static final String WORKLOGS_SUBDIR = "worklogs";
    }

    /**
     * Returns the root path of JIRA's indexes.
     * This directory is a sub-directory of the configured jira-home and is defined in the Atlassian home directory
     * specification to live under [jira-home]/caches/indexesV1/ by default.
     *
     * @return the root path of JIRA's indexes
     */
    String getIndexRootPath();

    /**
     * Returns the path of JIRA's issue indexes.
     *
     * @return the path of JIRA's issue indexes
     */
    String getIssueIndexPath();

    /**
     * Returns the path of JIRA's comment indexes.
     *
     * @return the path of JIRA's comment indexes
     */
    String getCommentIndexPath();

    /**
     * Returns the path of JIRA's worklog indexes
     *
     * @return the path of JIRA's worklog indexes
     */
    String getWorklogIndexPath();

    /**
     * Returns the path of JIRA's change history indexes.
     *
     * @return the path of JIRA's change history indexes
     */
    String getChangeHistoryIndexPath();

    /**
     * Returns the root path of JIRA's plugin indexes.
     * <p>
     * NOTE: Each Plugin should create a new directory under this path
     *
     * @return the root path of JIRA's plugin indexes
     */
    String getPluginIndexRootPath();

    /**
     * Returns the path of JIRA's shared entity indexes.
     *
     * @return the path of JIRA's shared entity indexes
     */
    String getSharedEntityIndexPath();

    /**
     * Implementation of {@link IndexPathManager} that uses the {@link ApplicationProperties} to get the current paths.
     * Nothing outside of this class should ever refer to the JIRA_PATH_INDEX ApplicationProperty. That property
     * needs to be interpreted properly when using JIRA Home.
     */
    public class PropertiesAdaptor implements IndexPathManager {
        private final JiraHome jiraHome;

        public PropertiesAdaptor(final JiraHome jiraHome) {
            this.jiraHome = jiraHome;
        }

        public String getIndexRootPath() {
            // http://extranet.atlassian.com/display/DEV/Product+home+directories+specification
            return new File(jiraHome.getLocalHomePath(), INDEXES_DIR).getAbsolutePath();
        }

        public String getIssueIndexPath() {
            return PathUtils.joinPaths(getIndexRootPath(), Directory.ISSUES_SUBDIR);
        }

        public String getCommentIndexPath() {
            return PathUtils.joinPaths(getIndexRootPath(), Directory.COMMENTS_SUBDIR);
        }

        public String getWorklogIndexPath() {
            return PathUtils.joinPaths(getIndexRootPath(), Directory.WORKLOGS_SUBDIR);
        }

        public String getChangeHistoryIndexPath() {
            return PathUtils.joinPaths(getIndexRootPath(), Directory.CHANGE_HISTORY_SUBDIR);
        }

        public String getSharedEntityIndexPath() {
            return PathUtils.joinPaths(getIndexRootPath(), Directory.ENTITIES_SUBDIR);
        }

        public String getPluginIndexRootPath() {
            return PathUtils.joinPaths(getIndexRootPath(), Directory.PLUGINS_SUBDIR);
        }
    }
}
