package com.atlassian.jira.bc.issue.fields;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nonnull.ReturnValuesAreNonnullByDefault;
import com.atlassian.jira.bc.ServiceOutcome;
import com.atlassian.jira.bc.ServiceResult;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.List;

/**
 * @since 8.16
 */
@Internal
@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
public interface FieldConfigSchemeService {

    /**
     * @param user           User performing operation.
     * @param fieldId        Field id.
     * @param configSchemeId Id of field config scheme to get.
     * @return Service outcome containing data of field config scheme.
     */
    ServiceOutcome<FieldConfigScheme> getFieldConfigScheme(@Nonnull ApplicationUser user, String fieldId, long configSchemeId);

    /**
     * @param user    User performing operation.
     * @param fieldId Field id
     * @return Service outcome containing field config schemes for field.
     */
    ServiceOutcome<List<FieldConfigScheme>> getFieldConfigSchemesForField(@Nonnull ApplicationUser user, String fieldId);

    /**
     * Creates new field configuration scheme.
     *
     * @param user       User performing operation.
     * @param parameters Data of field config scheme to create.
     * @return service outcome containing data of created field config scheme.
     */
    ServiceOutcome<FieldConfigScheme> createFieldConfigScheme(@Nonnull ApplicationUser user, FieldConfigSchemeParameters parameters);

    /**
     * Updates field configuration scheme,
     *
     * @param user       User performing operation.
     * @param parameters Target data of field config scheme to update.
     * @return service outcome containing data of updated field config scheme.
     */
    ServiceOutcome<FieldConfigScheme> updateFieldConfigScheme(@Nonnull ApplicationUser user, FieldConfigSchemeParameters parameters);

    /**
     * Deletes field configuration scheme,
     *
     * @param loggedInUser   User performing operation.
     * @param fieldId        Field id of field associated with configuration scheme to delete
     * @param configSchemeId Id of configuration scheme to delete
     * @return outcome of operation.
     */
    ServiceResult deleteFieldConfigScheme(@Nonnull ApplicationUser loggedInUser, String fieldId, long configSchemeId);

    /**
     * Gets list of available projects that can be used for updating existing field context for passed {@code fieldId} and {@code configSchemeId}.
     *
     * @param loggedInUser   User performing operation.
     * @param fieldId        Field id
     * @param configSchemeId
     * @return list of available projects that can be used for updating existing field context for passed {@code fieldId} and {@code configSchemeId}.
     */
    ServiceOutcome<List<Project>> getAvailableProjectsForUpdate(@Nonnull ApplicationUser loggedInUser, String fieldId, long configSchemeId);

    /**
     * Checks if existing field context for passed {@code fieldId} and {@code configSchemeId} can be configured as "global" context.
     * Global context for update operation should be available when
     * either no global context is configured for any field config scheme associated with field
     * or updated field context scheme is configured as global scheme.
     *
     * @param loggedInUser   User performing operation.
     * @param fieldId        Field id
     * @param configSchemeId
     * @return true if existing field context for passed {@code fieldId} and {@code configSchemeId} can be configured as "global" context, false otherwise.
     */
    ServiceOutcome<Boolean> isGlobalAvailableForUpdate(@Nonnull ApplicationUser loggedInUser, String fieldId, long configSchemeId);

    /**
     * Gets list of available projects that can be used for new field context for passed {@code fieldId}.
     *
     * @param loggedInUser User performing operation.
     * @param fieldId      Field id
     * @return list of available projects that can be used for new field context for passed {@code fieldId}.
     */
    ServiceOutcome<List<Project>> getAvailableProjectsForCreate(@Nonnull ApplicationUser loggedInUser, String fieldId);

    /**
     * Checks if new field context for passed {@code fieldId} can be configured as "global" context.
     * Global context for create is available when no global context is configured for any field config scheme associated with field.
     *
     * @param loggedInUser User performing operation.
     * @param fieldId      Field id
     * @return true if new field context for passed {@code fieldId} can be configured as "global" context, false otherwise.
     */
    ServiceOutcome<Boolean> isGlobalAvailableForCreate(@Nonnull ApplicationUser loggedInUser, String fieldId);
}
