package com.atlassian.jira.util;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;

/**
 * Provides version information about the currently running build of Jira platform.
 *
 * @since v4.0
 */
@PublicApi
public interface BuildUtilsInfo {

    /**
     * Gets the current version of Jira
     *
     * @return the current version of Jira
     */
    String getVersion();

    /**
     * Return the version used for Jira documentation.
     * <p>
     * This is the version that Jira uses when generating links to its externally hosted documentation.
     *
     * @return the version used for Jira documentation.
     * @since 6.2.3.
     */
    String getDocVersion();

    /**
     * Gets the numerical components of the version.
     * <p>
     * <li>
     * <ul><code>"5.0"</code> gives [5, 0, 0]</ul>
     * <ul><code>"5.0.1"</code> gives [5, 0, 1]</ul>
     * <ul><code>"5.0-beta1"</code> gives [5, 0, 0]</ul>
     * <ul><code>"5.0.1-SNAPSHOT"</code> gives [5, 0, 1]</ul>
     * </li>
     * <p>
     * The returned array is guaranteed to have at least 3 elements. Any non-numeric suffix in getVersion() is ignored.
     * <p>
     * A simple way to use this information is with Guava's {@link com.google.common.primitives.Ints#lexicographicalComparator()}:
     * </p>
     * <pre>
     *  int[] v510 = {5, 1, 0};
     *  if (Ints.lexicographicalComparator().compare(buildUtils.getVersionNumbers(), v510) >= 0 ) { ...
     * </pre>
     *
     * @return the leading numerical components of getVersion()
     */
    int[] getVersionNumbers();

    /**
     * Gets the current build number of Jira.
     * <p>
     * This is the same value as {@link #getApplicationBuildNumber()} except it is returned as a String.
     *
     * @return the current build number of Jira
     * @see #getDatabaseBuildNumber()
     * @see #getApplicationBuildNumber()
     */
    String getCurrentBuildNumber();

    /**
     * Gets the build number of this Jira install.
     * <p>
     * That is, the build number of the currently running installation files.
     * The DB also stores a build number.
     *
     * @return the build number of this Jira install.
     * @see #getDatabaseBuildNumber()
     */
    int getApplicationBuildNumber();

    /**
     * Gets the build number of the database that this Jira instance points to.
     * <p>
     * Under normal circumstances this will be the same as the build number of the Jira installation.
     * However, when you first upgrade a Jira database or import data exported from an older Jira, the existing database
     * will start on the previous build number. Jira will then run "Upgrade Tasks" to update the data in the DB.
     *
     * @return the build number of the database that Jira points to.
     * @see #getCurrentBuildNumber()
     */
    int getDatabaseBuildNumber();

    /**
     * Gets the minimal build number that Jira can upgrade from
     *
     * @return the minimal build number that Jira can upgrade from
     */
    String getMinimumUpgradableBuildNumber();

    /**
     * Gets the date this version of Jira was built on.
     *
     * @return the date this version of Jira was built on, or <code>null</code> if it couldn't be parsed
     */
    @Nullable
    Date getCurrentBuildDate();

    /**
     * Gets the partner name of this Jira build
     *
     * @return the partner name of this Jira build.
     */
    String getBuildPartnerName();

    /**
     * Gets a build information summary as a String.
     *
     * @return a build information summary
     */
    String getBuildInformation();

    /**
     * Returns the SCM commit id that was used to make this build of Jira. This used to be an SVN revision, but is now a
     * Git commit id (aka a SHA1). This method returns an empty string if Jira was built from the source distribution.
     *
     * @return a String containing the SCM commit id, or an empty String.
     * @deprecated Use {@link #getCommitId()} instead. Since v5.1.
     */
    @Deprecated
    String getSvnRevision();

    /**
     * Returns the id of the SCM commit that was used to make this build of Jira. This method returns an empty string if
     * Jira was built from the source distribution.
     *
     * @return the SCM commit id that was used to make this build of Jira.
     */
    String getCommitId();

    /**
     * Get the minimum version of Jira that can be upgraded to this instance version.
     *
     * @return the minimum version that can be upgraded.
     */
    String getMinimumUpgradableVersion();

    Collection<Locale> getUnavailableLocales();

    /**
     * Returns the version of Atlassian SAL that Jira exports into OSGI-land.
     *
     * @return the version of Atlassian SAL that Jira exports
     */
    String getSalVersion();

    /**
     * Returns the version of AppLinks that Jira ships with.
     *
     * @return the version of AppLinks that Jira ships with
     */
    String getApplinksVersion();

    /**
     * @return the osgi version that Guava (com.google.common) should be exported as
     */
    String getGuavaOsgiVersion();

    /**
     * @return the osgi version for the embedded crowd to be exported with
     */
    String getCrowdOsgiVersion();

    /**
     * Returns the value of the build property with the given key.
     *
     * @param key the key for which to retrieve the value
     * @return <code>null</code> if the key is unknown
     */
    @Nullable
    String getBuildProperty(String key);

    /**
     * Returns whether the current version is a beta (i.e. has -beta after its version numbers)
     *
     * @return true if the version string has -beta after its version numbers, otherwise false (case-insensitive)
     */
    boolean isBeta();

    /**
     * Indicates whether the current version is an Early Access Program (EAP) release.
     *
     * @return see above
     * @since 7.4
     */
    boolean isEap();

    /**
     * Indicates whether the current version is a final (GA) release.
     *
     * @return see above
     * @since 7.4
     */
    boolean isFinal();

    /**
     * Returns whether the current version is a release candidate (i.e. has -rc after its version numbers)
     *
     * @return true if the version string has -rc after its version numbers, otherwise false (case-insensitive)
     */
    boolean isRc();

    /**
     * Returns whether the current version is a snapshot (i.e. has -SNAPSHOT after its version numbers)
     *
     * @return true if the version string has -SNAPSHOT after its version numbers, otherwise false (case-insensitive)
     */
    boolean isSnapshot();

    /**
     * Returns whether the current version is a milestone (i.e. has -mXX after its version numbers)
     *
     * @return true if the version string has -m after its version numbers, otherwise false (case-insensitive)
     */
    boolean isMilestone();

    /**
     * Returns the version of Atlassian Audit that Jira exports into OSGI-land.
     *
     * @return the version of Atlassian Audit that Jira exports
     * @since 8.7
     */
    @ExperimentalApi
    String getAuditVersion();
}
