package com.atlassian.jira.project.archiving;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.bc.project.ProjectService;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;

import javax.annotation.Nonnull;

/**
 *  This service can be used for archiving projects and restoring projects.
 *
 *  @since 7.9
 */
@ExperimentalApi
public interface ArchivedProjectService {
    /**
     * Mark the project as archived and perform operation of archiving including all dependencies.
     * This method will also trigger {@link com.atlassian.jira.event.project.ProjectArchivedEvent}.
     *
     * @param validation a request previously validated by
     *                {@link #validateArchiveProject(ApplicationUser, String)}.
     * @return the result of the operation.
     */
    ArchivingResult archiveProject(ValidationResult validation);

    /**
     * Check if given project can be archived by given user.
     *
     * @param user
     * @param projectKey
     * @return the validation result.
     */
    ValidationResult validateArchiveProject(ApplicationUser user, String projectKey);

    /**
     * Mark the project as not archived.
     * This method will also trigger {@link com.atlassian.jira.event.project.ProjectRestoredEvent}.
     *
     * @param validation a request previously validated by
     *                {@link #validateRestoreProject(ApplicationUser, String)}.
     * @return the result of the operation.
     */
    ArchivingResult restoreProject(ValidationResult validation);

    /**
     * Check if given project can be restored by given user.
     *
     * @param user
     * @param projectKey
     * @return the validation result.
     */
    ValidationResult validateRestoreProject(ApplicationUser user, String projectKey);

    /**
     * A simple object that holds the information about an project archive/restore operation. This object should not be constructed
     * directly.
     */
    class ArchivingResult extends ValidationResult {
        public ArchivingResult(ErrorCollection errorCollection, @Nonnull Project project) {
            super(errorCollection, project);
        }
    }

    /**
     * A simple object that holds the information about validating an archive/restore project operation. This object should not be
     * constructed directly, you should invoke the {@link #validateArchiveProject(ApplicationUser, String)} or {@link #validateRestoreProject(ApplicationUser, String)} method
     * to obtain this.
     */
    class ValidationResult extends ProjectService.AbstractProjectResult {
        private final String projectKey;

        public ValidationResult(ErrorCollection errorCollection, String projectKey) {
            super(errorCollection, null);
            this.projectKey = projectKey;
        }

        public ValidationResult(ErrorCollection errorCollection, @Nonnull Project project) {
            super(errorCollection, project);
            this.projectKey = project.getKey();
        }

        public String getProjectKey() {
            return projectKey;
        }
    }
}
