package com.atlassian.jira.permission;

import com.atlassian.annotations.ExperimentalApi;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;

/**
 * Allows managing permission scheme attributes.
 *
 * @since 7.4
 */
@ExperimentalApi
@ParametersAreNonnullByDefault
public interface PermissionSchemeAttributeManager {

    class AttributeKeys {
        public static final String EXTENDED_ADMINISTER_PROJECTS_ENABLED_ATTRIBUTE = "ADMINISTER_PROJECTS.extended.enabled";
    }

    /**
     * Retrieves attribute value.
     * @param schemeId permission scheme id
     * @param key attribute key
     * @return {@link Optional Optional&lt;String&gt;} containing attribute value if it exists or default value if specified<br/>
     *         {@link Optional#empty() Optional&lt;String&gt;#empty()} if an attribute with the given key was not found for the given scheme
     *                                                                  and doesn't have specified default value
     */
    Optional<String> getAttribute(Long schemeId, String key);

    /**
     * Retrieves all attributes for the given permission scheme (including default attributes).
     * @param schemeId permission scheme id
     * @return {@link Map Map&lt;String, String&gt;} of attributes' keys and values for the given permission scheme
     */
    Map<String, String> getAttributes(Long schemeId);

    /**
     * Upserts an attribute for the given scheme.
     * @param schemeId permission scheme id
     * @param key attribute key
     * @param value attribute value
     */
    void setAttribute(Long schemeId, String key, String value);

    /**
     * Updates ALL attributes for the permission scheme.
     * @param schemeId permission scheme id
     * @param attributes {@link Map Map&lt;String, String&gt;} of all attributes the defined scheme should have,
     *                   e.g. if this map doesn't contain an existing attribute, it would be deleted
     */
    void updateAttributes(Long schemeId, Map<String, String> attributes);

    /**
     * Deletes attributes with the defined keys from the permission scheme.
     * @param schemeId permission scheme id
     * @param keys a {@link Collection Collection&lt;String&gt;} of attribute keys
     */
    void deleteAttributes(Long schemeId, Collection<String> keys);

    /**
     * Deletes all attributes from the permission scheme.
     * @param schemeId permission scheme id
     */
    void clearAttributes(Long schemeId);

    /**
     * Default attributes are meant to exist even if there's no such attribute stored.
     *
     * @return {@link Map Map&lt;String, String&gt;} of default attributes' keys and values
     */
    Map<String, String> getDefaultAttributes();
}
