package com.atlassian.jira.license;

import com.atlassian.application.api.ApplicationKey;
import com.atlassian.sal.api.validate.LicenseErrorCode;
import com.atlassian.sal.api.validate.LicenseValidationResult;
import com.atlassian.sal.api.validate.LicenseValidationWarning;
import com.atlassian.sal.api.validate.LicenseWarningCode;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.Set;

/**
 * A set of methods which describe the validation information of a specific license details of a specific {@link ApplicationKey}
 * of the currently installed license.
 *
 * @since v8.9
 */
public interface LicenseValidationInformation {

    /**
     * @return the {@link Optional} of the license detail.
     */
    Optional<LicenseDetails> getLicenseDetail();

    /**
     * @return the valid application key that the license is applied to
     */
    Optional<ApplicationKey> getApplicationKey();

    /**
     * @return the license validation result
     */
    LicenseValidationResult getLicenseValidationResult();

    /**
     * @return true if plan to removed {@link #getLicenseDetail()} out from the product, false otherwise.
     */
    boolean isDeleteLicense();

    /**
     * @return true if the value of {@link #getApplicationKey()} is valid, false otherwise.
     */
    boolean isProductKeyValid();

    /**
     * @return true if the {@link #getLicenseDetail()} is valid, false otherwise.
     */
    boolean isLicenseDetailsValid();

    /**
     * @return true if {@link #getLicenseDetail()} can apply to {@link #getApplicationKey()}, false otherwise.
     */
    boolean isLicensePairWithCorrectProductKey();

    /**
     * @return true if the {@link #getLicenseValidationResult()} is valid,
     *          false when the {@link #getLicenseDetail()} has failed at least one validation
     *              or {@link #isDeleteLicense()} is true.
     */
    boolean needMoreValidation();

    /**
     * @return a defensive copy of this LicenseValidationInformation with a giving {@link LicenseValidationResult}.
     */
    LicenseValidationInformation with(@Nonnull LicenseValidationResult licenseValidationResult);

    /**
     * @param licenseErrorCode the validation error code
     * @param errorMessage the error message
     * @return a defensive copy of this LicenseValidationInformation with error validation result
     */
    LicenseValidationInformation withError(@Nonnull LicenseErrorCode licenseErrorCode, @Nonnull String errorMessage);

    /**
     * @param licenseWarningCode the validation warning code
     * @param warningMessage the warning message
     * @return a defensive copy of this LicenseValidationInformatio with warning validation result
     */
    LicenseValidationInformation withWarning(@Nonnull LicenseWarningCode licenseWarningCode, @Nonnull String warningMessage);

    /**
     * @param warnings the list of warning validation results
     * @return a defensive copy of this LicenseValidationInformatio with warning validation result
     */
    LicenseValidationInformation withWarnings(@Nonnull Set<LicenseValidationWarning> warnings);
}
