package com.atlassian.jira.jql.validator;

import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.jira.util.MessageSet;
import com.atlassian.jira.util.MessageSetImpl;
import com.atlassian.query.clause.TerminalClause;
import com.atlassian.query.operand.EmptyOperand;
import com.atlassian.query.operand.FunctionOperand;
import com.atlassian.query.operand.MultiValueOperand;
import com.atlassian.query.operand.OperandVisitor;
import com.atlassian.query.operand.SingleValueOperand;
import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;

import static com.atlassian.jira.jql.validator.AbstractVersionValidator.LIKE_OPERATORS;
import static java.util.Objects.requireNonNull;

/**
 * Limits the number of wildcards in the operand.
 * <p>
 * Prevents bad RegExp backtracking, resulting in threads using 100% CPU, when using lots of wildcards in versions queries.
 *
 * @see AbstractVersionValidator
 * @since v8.12
 */
public class WildcardCountValidator {
    private final I18nHelper.BeanFactory beanFactory;

    public WildcardCountValidator(I18nHelper.BeanFactory beanFactory) {
        this.beanFactory = requireNonNull(beanFactory);
    }

    @Nonnull
    protected MessageSet validate(final ApplicationUser searcher, @Nonnull final TerminalClause terminalClause) {
        if (!LIKE_OPERATORS.contains(terminalClause.getOperator())) {
            return new MessageSetImpl();
        }

        final OperandVisitor<MessageSet> validator = new WildcardCountVisitor(beanFactory.getInstance(searcher));
        return terminalClause.getOperand().accept(validator);
    }

    @VisibleForTesting
    static class WildcardCountVisitor implements OperandVisitor<MessageSet> {
        private static final String WILDCARD = "*";
        private static final int WILDCARD_LIMIT = 1;

        private final I18nHelper i18n;

        WildcardCountVisitor(I18nHelper i18n) {
            this.i18n = requireNonNull(i18n);
        }

        @Override
        public MessageSet visit(EmptyOperand empty) {
            return new MessageSetImpl();
        }

        @Override
        public MessageSet visit(FunctionOperand function) {
            return new MessageSetImpl();
        }

        @Override
        public MessageSet visit(MultiValueOperand multiValue) {
            return new MessageSetImpl();
        }

        @Override
        public MessageSet visit(SingleValueOperand singleValue) {
            final MessageSet errors = new MessageSetImpl();

            final int wildcardsCount = StringUtils.countMatches(singleValue.getStringValue(), WILDCARD);
            if (wildcardsCount > WILDCARD_LIMIT) {
                errors.addErrorMessage(i18n.getText("jira.jql.clause.too.many.wildcards", new Object[]{WILDCARD_LIMIT, singleValue.getDisplayString()}));
            }

            return errors;
        }
    }
}
