package com.atlassian.jira.jql.validator;

import com.atlassian.jira.jql.operand.JqlOperandResolver;
import com.atlassian.jira.jql.operand.QueryLiteral;
import com.atlassian.jira.jql.query.QueryCreationContext;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.jira.util.ListOrderedMessageSetImpl;
import com.atlassian.jira.util.MessageSet;
import com.atlassian.jira.util.dbc.Assertions;
import com.atlassian.query.clause.TerminalClause;
import com.atlassian.query.operand.Operand;
import com.atlassian.query.operator.Operator;

import java.util.List;

/**
 * A clause validator that can be used for multiple constant (priority, status, resolution) clause types.
 */
abstract class ValuesExistValidator {
    private final JqlOperandResolver operandResolver;
    private final I18nHelper.BeanFactory beanFactory;
    private final MessageSet.Level level;


    ValuesExistValidator(final JqlOperandResolver operandResolver, final I18nHelper.BeanFactory beanFactory) {
        this(operandResolver, beanFactory, MessageSet.Level.ERROR);
    }

    ValuesExistValidator(final JqlOperandResolver operandResolver, final I18nHelper.BeanFactory beanFactory, final MessageSet.Level level) {
        this.beanFactory = Assertions.notNull("beanFactory", beanFactory);
        this.operandResolver = Assertions.notNull("operandResolver", operandResolver);
        this.level = level;
    }

    MessageSet validate(final TerminalClause terminalClause, final QueryCreationContext queryCreationContext) {
        final Operand operand = terminalClause.getOperand();
        final List<QueryLiteral> rawOperandValues = operandResolver.getValues(queryCreationContext, operand, terminalClause);

        return validate(queryCreationContext.getApplicationUser(), terminalClause, rawOperandValues);
    }

    MessageSet validate(final ApplicationUser searcher, final TerminalClause terminalClause) {
        final Operand operand = terminalClause.getOperand();
        final List<QueryLiteral> rawOperandValues = operandResolver.getValues(searcher, operand, terminalClause);

        return validate(searcher, terminalClause, rawOperandValues);
    }

    private MessageSet validate(final ApplicationUser searcher, final TerminalClause terminalClause, final List<QueryLiteral> rawOperandValues) {

        final MessageSet messages = new ListOrderedMessageSetImpl();

        if (operandResolver.isValidOperand(terminalClause.getOperand())) {
            // visit every query literal and determine lookup failures
            for (QueryLiteral rawValue : rawOperandValues) {
                validateOperandValue(searcher, terminalClause, rawValue, messages);
            }
        }

        return messages;
    }

    private void validateOperandValue(ApplicationUser searcher, TerminalClause terminalClause, QueryLiteral rawOperandValue, MessageSet messages) {
        if (rawOperandValue.getStringValue() != null) {
            validateStringOperandValue(searcher, terminalClause, rawOperandValue, messages);
        } else if (rawOperandValue.getLongValue() != null) {
            validateLongOperandValue(searcher, terminalClause, rawOperandValue, messages);
        }
    }

    private void validateLongOperandValue(ApplicationUser searcher, TerminalClause terminalClause, QueryLiteral rawOperandValue, MessageSet messages) {
        if (!longValueExist(searcher, rawOperandValue.getLongValue())) {
            if (operandResolver.isFunctionOperand(rawOperandValue.getSourceOperand())) {
                messages.addMessage(this.level, getI18n(searcher).getText("jira.jql.clause.no.value.for.name.from.function", rawOperandValue.getSourceOperand().getName(), terminalClause.getName()));
            } else {
                messages.addMessage(this.level, getI18n(searcher).getText("jira.jql.clause.no.value.for.id", terminalClause.getName(), rawOperandValue.getLongValue().toString()));
            }
        }
    }

    private void validateStringOperandValue(ApplicationUser searcher, TerminalClause terminalClause, QueryLiteral rawOperandValue, MessageSet messages) {
        if (!stringValueExists(searcher, rawOperandValue.getStringValue())) {
            if (operandResolver.isFunctionOperand(rawOperandValue.getSourceOperand())) {
                messages.addMessage(this.level, getI18n(searcher).getText("jira.jql.clause.no.value.for.name.from.function", rawOperandValue.getSourceOperand().getName(), terminalClause.getName()));
            } else {
                messages.addMessage(this.level, getI18n(searcher).getText("jira.jql.clause.no.value.for.name", terminalClause.getName(), rawOperandValue.getStringValue()));
                if ( Operator.EQUALS.equals(terminalClause.getOperator()) && (rawOperandValue.getStringValue().contains("*") || rawOperandValue.getStringValue().contains("?"))) {
                    messages.addMessage(this.level, getI18n(searcher).getText("jira.jql.clause.use.contains.operator"));
                }
            }
        }
    }

    abstract boolean stringValueExists(final ApplicationUser searcher, String value);

    abstract boolean longValueExist(final ApplicationUser searcher, Long value);

    ///CLOVER:OFF
    I18nHelper getI18n(final ApplicationUser user) {
        return beanFactory.getInstance(user);
    }
    ///CLOVER:ON
}
