package com.atlassian.jira.jql.query;

import com.atlassian.jira.issue.customfields.converters.DoubleConverter;
import com.atlassian.jira.issue.index.DocumentConstants;
import com.atlassian.jira.jql.operand.QueryLiteral;
import com.atlassian.jira.jql.operator.OperatorClasses;
import com.atlassian.jira.lucenelegacy.NumericUtils;
import com.atlassian.query.operator.Operator;
import com.google.common.collect.Lists;
import org.apache.log4j.Logger;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;

import java.util.List;

/**
 * Creates equality queries for clauses whose value is exactly the same as the indexed value(e.g. votes and duration).
 *
 * @since v6.4
 */
public class NumberEqualityQueryFactory extends AbstractNumberOperatorQueryFactory
        implements OperatorSpecificQueryFactory {
    private static final Logger log = Logger.getLogger(NumberEqualityQueryFactory.class);

    /**
     * Creates a Query Factory that represents Empty values with emptyIndexValue
     *
     * @param doubleConverter used for converting query literals to the index representation
     * @param emptyIndexValue this value is ignored
     * @deprecated since 8.9 use {@link #NumberEqualityQueryFactory(DoubleConverter)}
     */
    public NumberEqualityQueryFactory(final DoubleConverter doubleConverter, final Double emptyIndexValue) {
        this(doubleConverter);
    }

    /**
     * Creates a Query Factory that does not have a specified representation of empty values and instead checks for the
     * field value absence.
     *
     * @param doubleConverter used for converting query literals to the index representation
     */
    public NumberEqualityQueryFactory(DoubleConverter doubleConverter) {
        super(doubleConverter);
    }

    public QueryFactoryResult createQueryForSingleValue(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        if (operator != Operator.EQUALS && operator != Operator.NOT_EQUALS) {
            log.debug(String.format("Creating an equality query for a single value for field '%s' using unsupported operator: '%s', returning "
                    + "a false result (no issues). Supported operators are: '%s' and '%s'", fieldName, operator, Operator.EQUALS, Operator.NOT_EQUALS));

            return QueryFactoryResult.createFalseResult();
        }
        return createResult(fieldName, operator, rawValues);
    }

    public QueryFactoryResult createQueryForMultipleValues(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        if (operator == Operator.IN || operator == Operator.NOT_IN) {
            return createResult(fieldName, operator, rawValues);
        } else {
            log.debug(String.format("Creating an equality query for multiple values for field '%s' using unsupported operator: '%s', returning "
                    + "a false result (no issues). Supported operators are: '%s' and '%s'", fieldName, operator, Operator.IN, Operator.NOT_IN));

            return QueryFactoryResult.createFalseResult();
        }
    }

    public QueryFactoryResult createQueryForEmptyOperand(final String fieldName, final Operator operator) {
        if (operator == Operator.IS || operator == Operator.EQUALS) {
            return new QueryFactoryResult(getIsEmptyQuery(fieldName));
        } else if (operator == Operator.IS_NOT || operator == Operator.NOT_EQUALS) {
            return new QueryFactoryResult(getIsNotEmptyQuery(fieldName));
        } else {
            log.debug(String.format("Creating an equality query for an empty value for field '%s' using unsupported operator: '%s', returning "
                            + "a false result (no issues). Supported operators are: '%s','%s', '%s' and '%s'", fieldName, operator,
                    Operator.IS, Operator.EQUALS, Operator.IS_NOT, Operator.NOT_EQUALS));

            return QueryFactoryResult.createFalseResult();
        }
    }

    public boolean handlesOperator(final Operator operator) {
        return OperatorClasses.EQUALITY_OPERATORS_WITH_EMPTY.contains(operator);
    }

    private QueryFactoryResult createResult(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        if (operator == Operator.IN || operator == Operator.EQUALS) {
            return handleIn(fieldName, getIndexValues(rawValues));
        } else if (operator == Operator.NOT_IN || operator == Operator.NOT_EQUALS) {
            return handleNotIn(fieldName, getIndexValues(rawValues));
        } else {
            return QueryFactoryResult.createFalseResult();
        }
    }

    /*
     * The IN operator is represented by a series of Equals clauses, ORed together.
     */
    private QueryFactoryResult handleIn(final String fieldName, final List<Double> values) {
        if (values.size() == 1) {
            final Double value = values.get(0);
            Query query = (value == null) ? getIsEmptyQuery(fieldName) : getTermQuery(fieldName, value);
            return new QueryFactoryResult(query);
        } else {
            BooleanQuery.Builder combined = new BooleanQuery.Builder();
            for (Double value : values) {
                if (value == null) {
                    combined.add(getIsEmptyQuery(fieldName), BooleanClause.Occur.SHOULD);
                } else {
                    combined.add(getTermQuery(fieldName, value), BooleanClause.Occur.SHOULD);
                }
            }
            return new QueryFactoryResult(combined.build());
        }
    }

    /*
     * The NOT IN operator is represented by a series of Not Equals clauses, ANDed together
     */
    private QueryFactoryResult handleNotIn(final String fieldName, final List<Double> values) {
        final List<Query> notQueries = Lists.newArrayListWithCapacity(values.size());

        for (Double indexValue : values) {
            // don't bother keeping track of empty literals - empty query gets added later anyway
            if (indexValue != null) {
                notQueries.add(getTermQuery(fieldName, indexValue));
            }
        }

        if (notQueries.isEmpty()) {
            // if we didn't find non-empty literals, then return the isNotEmpty query
            return new QueryFactoryResult(getIsNotEmptyQuery(fieldName));
        } else {
            BooleanQuery.Builder boolQuery = new BooleanQuery.Builder();
            // Because this is a NOT equality query we are generating we always need to explicity exclude the
            // EMPTY results from the query we are generating.
            boolQuery.add(getIsNotEmptyQuery(fieldName), BooleanClause.Occur.FILTER);

            // Add all the not queries that were specified by the user.
            for (Query query : notQueries) {
                boolQuery.add(query, BooleanClause.Occur.MUST_NOT);
            }

            // We should add the visibility query so that we exclude documents which don't have fieldName indexed.
            boolQuery.add(TermQueryFactory.visibilityQuery(fieldName), BooleanClause.Occur.MUST);

            return new QueryFactoryResult(boolQuery.build(), false);
        }
    }

    private Query getIsEmptyQuery(final String fieldName) {
        // We are returning a query that will include empties by specifying a MUST_NOT occurrence.
        // We should add the visibility query so that we exclude documents which don't have fieldName indexed.
        QueryFactoryResult result = new QueryFactoryResult(TermQueryFactory.nonEmptyQuery(fieldName), true);
        return QueryFactoryResult.wrapWithVisibilityQuery(fieldName, result).getLuceneQuery();
    }

    private Query getIsNotEmptyQuery(final String fieldName) {
        return TermQueryFactory.nonEmptyQuery(fieldName);
    }

    private Query getTermQuery(final String fieldName, final Double value) {
        return new TermQuery(new Term(DocumentConstants.LUCENE_SORTFIELD_PREFIX + fieldName, NumericUtils.doubleToPrefixCoded(value)));
    }
}
