package com.atlassian.jira.jql.builder;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.isNullOrEmpty;

/**
 * Represents a JQL Field Reference which has a form of field_name[property].object_reference with
 * object_reference part being optional
 */
@ThreadSafe
public final class JqlFieldReference {
    private final String name;
    private final String property;
    private final String reference;

    JqlFieldReference(String name, String property) {
        this(name, property, null);
    }

    JqlFieldReference(String name, String property, String reference) {
        checkArgument(!isNullOrEmpty(name), "Field name is not supposed to be empty");
        checkArgument(!isNullOrEmpty(property), "Property is not supposed to be empty");

        this.name = name;
        this.property = property;
        this.reference = reference;
    }

    /**
     * The field name of a Field Reference
     *
     * @return String
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * The property of a Field Reference
     *
     * @return String
     */
    @Nonnull
    public String getProperty() {
        return property;
    }

    /**
     * Optional Object Reference of a Field Reference
     *
     * @return String
     */
    @Nullable
    public String getReference() {
        return reference;
    }

    /**
     * JQL string representation of a Field Reference
     *
     * @return
     */
    public String toJql() {
        if(hasReference()) {
            return String.format("%s[%s].%s", name, property, reference);
        } else {
            return String.format("%s[%s]", name, property);
        }
    }

    public static JqlFieldReferenceBuilder create() {
        return new JqlFieldReferenceBuilder();
    }

    /**
     * If Field Reference has an object reference.
     *
     * Because {@link com.atlassian.query.clause.Property} returns an empty string for empty list of references
     * we need to allow users to supply empty strings as reference and treat them as "no object references.
     *
     * @return true if field reference includes object reference for a property.
     */
    public boolean hasReference() {
        return !isNullOrEmpty(reference);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        JqlFieldReference that = (JqlFieldReference) o;

        if (!name.equals(that.name)) return false;
        if (!property.equals(that.property)) return false;
        return reference != null ? reference.equals(that.reference) : that.reference == null;
    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + property.hashCode();
        result = 31 * result + (reference != null ? reference.hashCode() : 0);
        return result;
    }
}
