package com.atlassian.jira.issue.search.util;

import com.google.common.collect.ImmutableSet;

import java.util.Set;

public class TextTermEscaper {

    /**
     * The full list of reserved chars in Lucene:
     * '\\', '+', '-', '!', '(', ')', ':', '^', '[', ']', '\"', '{', '}', '~', '*', '?', '|', '&', '/'
     * We split this into allowed and disallowed operators for user to use in JQL.
     *
     * This set denotes the allowed operators.
     */
    private static final Set<Character> ALLOWED_LUCENE_OPERATORS = ImmutableSet.of('+', '-', '&', '|', '!', '(', ')', '{', '}', '[', ']', '^', '~', '*', '?', '\"');

    /**
     * This set denotes the disallowed operators (as opposed to with {@link #ALLOWED_LUCENE_OPERATORS}).
     * User must not type in these characters to Lucene.
     * ':' is a field selector. This might give the user access to fields they are not allowed to see. See JRA-16151.
     * '/' marks a regex. We are not supporting regex searches, see JSEV-2450.
     */
    private static final Set<Character> DISALLOWED_LUCENE_OPERATORS = ImmutableSet.of(':', '/');

    public static String escape(final CharSequence input) {
        final StringBuilder escaped = new StringBuilder(input.length() * 2);
        for (int i = 0; i < input.length(); i++) {
            final Character c = input.charAt(i);
            // we always escape these:
            if (DISALLOWED_LUCENE_OPERATORS.contains(c)) {
                escaped.append('\\');
            } else if (c.equals('\\')) {
                final Character nextChar = i + 1 < input.length() ? input.charAt(i + 1) : null;

                if (shouldEscapeBackslash(nextChar)) {
                    escaped.append('\\');
                }
            }

            escaped.append(c);
        }
        return escaped.toString();
    }

    /**
     * Returns true if user must not escape the proceeding character. In such a case we escape the escape, making the backslash an ordinary character.
     */
    private static boolean shouldEscapeBackslash(final Character charAfterBackslash) {
        return !ALLOWED_LUCENE_OPERATORS.contains(charAfterBackslash);
    }
}
