package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.indexers.phrase.PhraseQuerySupportField;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.SortedDocValuesField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.TextField;
import org.apache.lucene.util.BytesRef;

import static com.atlassian.jira.util.dbc.Assertions.notBlank;
import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * A custom field indexer for text fields that can be sorted
 *
 * @since v4.0
 */
public class SortableTextCustomFieldIndexer extends AbstractCustomFieldIndexer {
    private final String sortFieldPrefix;
    private final boolean skipIndexingNull;

    public SortableTextCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager,
                                          final CustomField customField,
                                          final String sortFieldPrefix,
                                          final boolean skipIndexingNull) {
        super(fieldVisibilityManager, notNull("field", customField));
        this.sortFieldPrefix = notBlank("sortFieldPrefix", sortFieldPrefix);
        this.skipIndexingNull = skipIndexingNull;
    }

    public SortableTextCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager,
                                          final CustomField customField,
                                          final String sortFieldPrefix) {
        this(fieldVisibilityManager, customField, sortFieldPrefix, false);
    }

    public void addDocumentFieldsSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, true);
    }

    public void addDocumentFieldsNotSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, false);
    }

    @Override
    public Boolean skipsIndexingNull() {
        return skipIndexingNull;
    }

    private void addDocumentFields(final Document doc, final Issue issue, final boolean searchable) {
        final String value = (String) customField.getValue(issue);
        final String valueForSorting = FieldIndexerUtil.getValueForSorting(value);
        if (value != null) {
            if (searchable) {
                doc.add(new TextField(getDocumentFieldId(), value, Field.Store.YES));
                doc.add(new TextField(PhraseQuerySupportField.forIndexField(getDocumentFieldId()), value, Field.Store.YES));
                doc.add(new SortedDocValuesField(sortFieldPrefix + getDocumentFieldId(), new BytesRef(valueForSorting)));
            } else {
                doc.add(new StoredField(getDocumentFieldId(), value));
                doc.add(new StoredField(PhraseQuerySupportField.forIndexField(getDocumentFieldId()), value));
            }
        } else {
            if (searchable && !skipIndexingNull) {
                doc.add(new SortedDocValuesField(sortFieldPrefix + getDocumentFieldId(), new BytesRef(valueForSorting)));
            }
        }
    }
}