package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.annotations.Internal;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.SortedDocValuesField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.util.BytesRef;

/**
 * Used for doing simple indexing stuff.
 *
 * @since v4.0
 */
@Internal
public class FieldIndexerUtil {
    private static final int MAX_SORT_LENGTH = 50;

    /**
     * Index a single keyword field, with a default if the issue field is not set
     * <p>
     * shared with CommentDocument
     */
    public static void indexKeywordWithDefault(final Document doc, final String indexField, final String fieldValue, final String defaultValue) {
        indexKeywordWithDefault(doc, indexField, fieldValue, defaultValue, true);
    }

    public static void indexKeywordWithDefault(final Document doc, final String indexField, final Long aLong, final String defaultValue) {
        final String value = aLong != null ? aLong.toString() : null;
        indexKeywordWithDefault(doc, indexField, value, defaultValue);
    }

    public static void indexKeywordWithDefault(final Document doc, final String indexField, final String fieldValue, final String defaultValue, final boolean searchable) {
        final String value = getOrDefault(fieldValue, defaultValue);
        if (searchable) {
            doc.add(new StringField(indexField, value, Field.Store.YES));
            doc.add(new SortedDocValuesField(indexField, new BytesRef(value)));
        } else {
            doc.add(new StoredField(indexField, value));
        }
    }

    public static String getOrDefault(String fieldValue, String defaultValue) {
        return (StringUtils.isNotBlank(fieldValue)) ? fieldValue : defaultValue;
    }

    public static String getValueForSorting(final String fieldValue) {
        final String trimmed = (fieldValue == null) ? null : fieldValue.trim();
        if (!StringUtils.isBlank(trimmed)) {
            if (trimmed.length() > MAX_SORT_LENGTH) {
                return trimmed.substring(0, MAX_SORT_LENGTH);
            }
            return trimmed;
        } else {

            return String.valueOf('\ufffd');
        }
    }
}
