package com.atlassian.jira.issue.index.indexers;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicSpi;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.CustomFieldType;
import com.atlassian.jira.issue.customfields.vdi.CustomFieldPrefetchedData;
import com.atlassian.jira.issue.customfields.vdi.NonNullCustomFieldProvider;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.FieldType;

import java.util.List;

/**
 * This is meant to create a portion of a {@link org.apache.lucene.document.Document} that is relevant for a JIRA
 * field.
 * <p>
 * The portion that is added to the Document will be the indexed value that is represented by the value contained
 * in the Issue object when the {@link #addIndex(Document, Issue, CustomFieldPrefetchedData)}
 * method is called.
 * <p>
 * There is a strong relationship between a FieldIndexer and a {@link com.atlassian.jira.jql.query.ClauseQueryFactory}
 * as the indexer creates a portion of the lucene Document and the query factory assumes it knows what this looks
 * like and can generate a Lucene query that will find relevant issues based on the values that were indexed.
 * <p>
 * If you are writing a CustomFieldSearcher then the glue that binds the FieldIndexer and the ClauseQueryFactory
 * together is the {@link com.atlassian.jira.issue.customfields.CustomFieldSearcher} since it provides both
 * the ClauseHandler and the FieldIndexer. Keep in mind that if you are creating one of these that the Indexer
 * must add fields to the document that the ClauseQueryFactory knows how to search.
 */
@PublicSpi
public interface FieldIndexer {
    /**
     * General empty token
     */
    String NO_VALUE_INDEX_VALUE = "-1";
    /**
     * Empty token specific to LabelsIndexer
     */
    String LABELS_NO_VALUE_INDEX_VALUE = "<EMPTY>";

    /**
     * @return the String representation of the field id that this indexer is indexing, this must be unique for
     * each independent FieldIndexer. If the Indexer does not represent a System or Custom field in JIRA this
     * should still return a unique string that describes the indexer.
     */
    String getId();

    /**
     * @return the String representation of the primary field id that will be added to the
     * {@link org.apache.lucene.document.Document} as a result of a successful call to the
     * {@link #addIndex(Document, Issue, CustomFieldPrefetchedData)} method.
     */
    String getDocumentFieldId();

    /**
     * This method allows an indexer the opportunity to modify the provided Lucene document (by reference) such
     * that it will contain fields that are relevant for searching and storage of the portion of the issue that
     * the FieldIndexer handles.
     * <p>
     * If calling {@link #isFieldVisibleAndInScope(com.atlassian.jira.issue.Issue)} returns false then
     * this method should create fields that have an Indexed type of {@link org.apache.lucene.document.Field.Index#NO}.
     * This allows us to store the value in the index but renders its value unsearchable.
     * <p>
     * If, for example, the indexer handles indexing an issues summary then this indexer will add a field to
     * the document that represents the stored and searchable summary of the issue.
     * <p>
     * Note that trying to index very large (i.e. larger than ~32 kB) fields as StringFields or DocValues will result
     * in them being removed to prevent the indexing operation from crashing. This is a limitation of how Lucene stores
     * terms and DocValues.
     *
     * @param doc   the lucene document that should be modified by adding relevant fields to.
     * @param issue the issue that contains the data that will be indexed and which can be used to determine
     *              the project/issue type context that will allow you to determine if we should add the value as searchable
     *              or unsearchable.
     * @deprecated since 8.10 use {@link #addIndex(Document, Issue, CustomFieldPrefetchedData)} instead
     */
    @Deprecated
    default void addIndex(final Document doc, final Issue issue) {
        throw new UnsupportedOperationException();
    }

    /**
     * @deprecated since 8.12 use {@link #addIndex(Document, Issue, CustomFieldPrefetchedData)} instead
     */
    @Deprecated
    default void addIndex(final Document doc, final Issue issue, final com.atlassian.jira.issue.customfields.CustomFieldPrefetchedData prefetchedData) {
        addIndex(doc, issue);
    }

    /**
     * This method allows an indexer the opportunity to modify the provided Lucene document (by reference) such
     * that it will contain fields that are relevant for searching and storage of the portion of the issue that
     * the FieldIndexer handles.
     * <p>
     * If calling {@link #isFieldVisibleAndInScope(com.atlassian.jira.issue.Issue)} returns false then
     * this method should create fields that have {@link FieldType#stored()} } property set to true.
     * This allows us to store the value in the index but renders its value unsearchable.
     * <p>
     * If, for example, the indexer handles indexing an issues summary then this indexer will add a field to
     * the document that represents the stored and searchable summary of the issue.
     * <p>
     * Note that trying to index very large (i.e. larger than ~32 kB) fields as StringFields or DocValues will result
     * in them being removed to prevent the indexing operation from crashing. This is a limitation of how Lucene stores
     * terms and DocValues.
     *  @param doc   the Lucene document that should be modified by adding relevant fields to.
     * @param issue the issue that contains the data that will be indexed and which can be used to determine
     *              the project/issue type context that will allow you to determine if we should add the value as searchable
     *              or unsearchable.
     * @param prefetchedData if the custom field type implemented {@link CustomFieldType#getNonNullCustomFieldProvider()}
 *              this reference will hold the matching value returned from {@link NonNullCustomFieldProvider#getCustomFieldInfo(List <Issue>)}
     */
    default void addIndex(final Document doc, final Issue issue, final CustomFieldPrefetchedData prefetchedData) {
        addIndex(doc, issue);
    }

    /**
     * This method is used to determine if the indexer is relevant for the provided issue. This method must check
     * the fields visibility, in relation to the field configuration scheme, must check any global flags that would
     * enable or disable a field (such as enable votes flag), and must check, if the field is a custom field, if
     * the custom field is relevant for this issue.
     * <p>
     * All these checks should take into account the {@link com.atlassian.jira.issue.context.IssueContext} as defined by
     * the passed in issue.
     * <p>
     * If this method returns false then the FieldIndexer, when performing addIndex, should make sure to make the
     * indexed values have an Indexed type of {@link org.apache.lucene.document.Field.Index#NO}.
     * <p>
     * The result of this method is used to determine the correct values that should be returned when performing
     * an empty search.
     *
     * @param issue that is having a document created from.
     * @return if true then this field is relevant for the issue, otherwise it is not.
     */
    boolean isFieldVisibleAndInScope(final Issue issue);

    /**
     * Used to declare how the indexer handles nulls.
     *
     * @return true iff the indexer doesn't do anything when the field value is null
     *
     * @since 8.10
     */
    @ExperimentalApi
    default Boolean skipsIndexingNull() {
        return null;
    }
}
