package com.atlassian.jira.issue.index;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.entity.WithId;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.comments.Comment;
import com.atlassian.jira.issue.util.IssuesIterable;
import com.atlassian.jira.issue.worklog.Worklog;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.task.context.Context;
import com.atlassian.jira.util.InjectableComponent;

import java.util.Collection;
import java.util.Set;

/**
 * Provides services related to the indexing of issue data.
 * This service largely replaces the {@code IssueIndexManager} that previously provided
 * those services.
 *
 * @since v7.0
 */
@InjectableComponent
@PublicApi
public interface IssueIndexingService {
    /**
     * Reindex all issues.
     *
     * @return Reindex time in ms.
     */
    long reIndexAll() throws IndexException;

    /**
     * Reindex all issues.
     *
     * @param context                 used to report progress back to the user or to the logs. Must not be null
     * @param useBackgroundReindexing whether to index in the background or not. If the useBackgroundReindexing option
     *                                is set to true, then all related fields will not be reindexed.
     * @param notifyCluster           whether or not to notify the cluster about reindexing all
     * @return Reindex time in ms.
     */
    long reIndexAll(final Context context, final boolean useBackgroundReindexing, final boolean notifyCluster) throws IndexException;

    /**
     * Reindex all issues. If the useBackgroundReindexing option is set to true, then only the basic issue information
     * will be reindexed, unless the indexing parameters are also set. This is considered the normal mode for background
     * re-indexing and is sufficient to correct the index for changes in the system configuration, but not for changes
     * to the indexing language. If useBackgroundReindexing is set to false, than everything is always reindexed.
     *
     * @param context                 used to report progress back to the user or to the logs. Must not be null
     * @param useBackgroundReindexing whether to index in the background or not
     * @param issueIndexingParams     determines witch related objects should be indexed together with issues. Only relevant
     *                                for background reindex operations.
     * @param notifyCluster           whether or not to notify the cluster about reindexing all
     * @return Reindex time in ms.
     */
    long reIndexAll(final Context context, final boolean useBackgroundReindexing, final IssueIndexingParams issueIndexingParams,
                    final boolean notifyCluster) throws IndexException;

    /**
     * Reindex a list of issues, passing an optional event that will be set progress
     *
     * @param issuesIterable IssuesIterable
     * @param context        used to report progress back to the user or to the logs. Must not be null.
     * @return Reindex time in ms.
     */
    long reIndexIssues(final IssuesIterable issuesIterable, final Context context) throws IndexException;

    /**
     * Reindex a list of issues, passing an optional event that will be set progress. This method can optionally also
     * index the comments and change history.
     *
     * @param issuesIterable      IssuesIterable
     * @param context             used to report progress back to the user or to the logs. Must not be null.
     * @param issueIndexingParams determines witch related objects should be indexed together with issue.
     * @return Reindex time in ms.
     */
    long reIndexIssues(final IssuesIterable issuesIterable, final Context context, final IssueIndexingParams issueIndexingParams)
            throws IndexException;

    /**
     * Reindex a list of issues, passing an optional Context for providing status updates. This method can optionally also
     * index the comments and change history. It also allows the caller to specify whether or not the indexing changes
     * should be replicated to the rest of the cluster.
     *
     * @param issuesIterable      IssuesIterable of issues to reindex
     * @param context             used to report progress back to the user or to the logs. Must not be null.
     * @param issueIndexingParams determines witch related objects should be indexed together with issue.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     * @return Reindex time in ms, -1 if there's an error.
     */
    long reIndexIssues(IssuesIterable issuesIterable, Context context, IssueIndexingParams issueIndexingParams, boolean shouldReplicate) throws IndexException;

    /**
     * Reindex an issue (eg. after field updates).
     */
    void reIndex(final Issue issue) throws IndexException;

    /**
     * Reindex an issue (eg. after field updates).
     */
    void reIndex(final Issue issue, final IssueIndexingParams issueIndexingParams) throws IndexException;

    /**
     * Reindexes a collection of comments.
     *
     * @param comments a collection of Comment
     */
    long reIndexComments(final Collection<Comment> comments) throws IndexException;

    /**
     * Reindexes a collection of comments.
     *
     * @param comments a collection of Comment
     * @param context  used to report progress back to the user or to the logs. Must not be null.
     */
    long reIndexComments(final Collection<Comment> comments, final Context context) throws IndexException;

    /**
     * Reindexes a collection of comments.
     *
     * @param comments        a collection of Comment
     * @param context         used to report progress back to the user or to the logs. Must not be null.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    long reIndexComments(final Collection<Comment> comments, final Context context, boolean shouldReplicate)
            throws IndexException;

    /**
     * Reindexes a collection of worklogs.
     *
     * @param worklogs a collection of Worklogs
     */
    long reIndexWorklogs(final Collection<Worklog> worklogs) throws IndexException;

    /**
     * Reindexes a collection of worklogs.
     *
     * @param worklogs a collection of Worklogs
     * @param context  used to report progress back to the user or to the logs. Must not be null.
     */
    long reIndexWorklogs(final Collection<Worklog> worklogs, final Context context) throws IndexException;

    /**
     * Reindexes a collection of worklogs.
     *
     * @param worklogs        a collection of Worklogs
     * @param context         used to report progress back to the user or to the logs. Must not be null.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    long reIndexWorklogs(final Collection<Worklog> worklogs, final Context context, boolean shouldReplicate)
            throws IndexException;

    /**
     * Remove an issue from the search index.
     */
    void deIndex(final WithId issueWithId) throws IndexException;

    /**
     * Remove an issue from the search index. Deprecated--kept for binary compatibility. Remove in 9.0
     */
    @Deprecated //remove in 9.0
    default void deIndex(final Issue issue) throws IndexException {
        deIndex((WithId)issue);
    }

    /**
     * Remove a set of issues from the search index.
     *
     * @param issuesToDelete set of {@link WithId}s to deIndex.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    void deIndexIssueObjectsById(final Set<? extends WithId> issuesToDelete, final boolean shouldReplicate) throws IndexException;

    /**
     * Remove a set of issues from the search index. Deprecated--kept for binary compatibility. Change to take WithIds in 9.0
     *
     * @param issuesToDelete set of {@link com.atlassian.jira.issue.Issue}s to deIndex.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    @Deprecated //change to WithId in 9.0
    default void deIndexIssueObjects(final Set<Issue> issuesToDelete, final boolean shouldReplicate) throws IndexException {
        deIndexIssueObjectsById(issuesToDelete, shouldReplicate);
    }

    /**
     * Reindex a set of issues.
     *
     * @param issueObjects Set of {@link com.atlassian.jira.issue.Issue}s to reindex.
     * @return Reindex time in ms.
     */
    long reIndexIssueObjects(final Collection<? extends Issue> issueObjects) throws IndexException;

    /**
     * Reindex a set of issues.
     *
     * @param issueObjects        Set of {@link com.atlassian.jira.issue.Issue}s to reindex.
     * @param issueIndexingParams Determines witch related objects should be indexed together with issues.
     * @return Reindex time in ms.
     */
    long reIndexIssueObjects(final Collection<? extends Issue> issueObjects, final IssueIndexingParams issueIndexingParams)
            throws IndexException;

    /**
     * Reindex a set of issues.
     *
     * @param issueObjects        Set of {@link com.atlassian.jira.issue.Issue}s to reindex.
     * @param issueIndexingParams Determines witch related objects should be indexed together with issues.
     * @param shouldReplicate     whether or not the index changes should be replicated to the other nodes in the cluster.
     * @return Reindex time in ms.
     */
    long reIndexIssueObjects(final Collection<? extends Issue> issueObjects, final IssueIndexingParams issueIndexingParams,
                             boolean shouldReplicate) throws IndexException;

    /**
     * Deindexing issues from a given project
     * @param project         project object
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    void deIndex(final Project project, final boolean shouldReplicate);

    /**
     * Remove a set of comments from the search index.
     *
     * @param commentsToDelete set of {@link WithId}s to deIndex.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    void deIndexComments(Set<WithId> commentsToDelete, boolean shouldReplicate);

    /**
     * Remove a set of worklogs from the search index.
     *
     * @param worklogsToDelete set of {@link WithId}s to deIndex.
     * @param shouldReplicate whether or not the index changes should be replicated to the other nodes in the cluster.
     */
    void deIndexWorklogs(Set<WithId> worklogsToDelete, boolean shouldReplicate);
}
