package com.atlassian.jira.issue.fields.config.manager;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.event.issue.field.config.manager.PrioritySchemeCreatedEvent;
import com.atlassian.jira.event.issue.field.config.manager.PrioritySchemeDeletedEvent;
import com.atlassian.jira.event.issue.field.config.manager.PrioritySchemeUpdatedEvent;
import com.atlassian.jira.issue.context.IssueContext;
import com.atlassian.jira.issue.fields.config.FieldConfig;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.issue.priority.Priority;
import com.atlassian.jira.project.Project;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * Manager {@link FieldConfigScheme} for managing priority schemes.
 *
 * Example usage:
 * <blockquote><pre>{@code
 * PrioritySchemeManager prioritySchemeManager;
 * ProjectManager projectManager;
 * //creates a new FieldConfigScheme with 3 priorities assigned
 * FieldConfigScheme fieldConfigScheme = prioritySchemeManager.createWithDefaultMapping("Some name", "Some description", ImmutableList.of("1", "2", "3"));
 * //options/priorities are assigned to a particular FieldConfig
 * FieldConfig fieldConfig = prioritySchemeManager.getFieldConfigForDefaultMapping(fieldConfigScheme);
 * //sets a priority with ID=1 as the default priority
 * prioritySchemeManager.setDefaultOption(fieldConfig, "1");
 * //assigns a priority scheme to a "TEST" project
 * Project project = projectManager.getProjectObjByKey("TEST");
 * prioritySchemeManager.assignProject(fieldConfigScheme, project);
 *
 * //retrieves all priorities for a given priority scheme
 * Collection<Priority> priorities = prioritySchemeManager.getPrioritiesFromIds(prioritySchemeManager.getOptions(fieldConfig));
 * }</pre></blockquote>
 */
@ExperimentalApi
public interface PrioritySchemeManager {

    /**
     * Creates a new {@link FieldConfigScheme} with a default mapping to {@link FieldConfig}.
     * Also creates options in {@code OPTION_CONFIGURATION} table.
     * Triggers {@link PrioritySchemeCreatedEvent}.
     *
     * @param name        Scheme name
     * @param description Scheme description
     * @param optionIds   Collection of priority ids
     * @return created FieldConfigScheme
     */
    @Nonnull
    FieldConfigScheme createWithDefaultMapping(@Nonnull String name, @Nullable String description, @Nullable List<String> optionIds);

    /**
     * Updates {@link FieldConfigScheme} with provided data and options (priority IDs).
     * Default {@link FieldConfig} for {@link FieldConfigScheme} is used.
     * Triggers {@link PrioritySchemeUpdatedEvent}.
     *
     * @param fieldConfigScheme Field config scheme to update
     * @param optionIds         Collection of priority ids
     * @return updated FieldConfigScheme
     */
    FieldConfigScheme updateWithDefaultMapping(@Nonnull FieldConfigScheme fieldConfigScheme, @Nullable List<String> optionIds);

    /**
     * Retrieves {@link FieldConfig} that is used for a default mapping for a passed {@code fieldConfigScheme}.
     *
     * @param fieldConfigScheme
     * @return {@link FieldConfig} for default mapping for {@code fieldConfigScheme}
     */
    @Nullable
    FieldConfig getFieldConfigForDefaultMapping(@Nullable FieldConfigScheme fieldConfigScheme);

    /**
     * Removes {@link FieldConfigScheme}.
     * Triggers {@link PrioritySchemeDeletedEvent}.
     *
     * @param fieldConfigScheme {@link FieldConfigScheme} to delete
     */
    void delete(@Nonnull FieldConfigScheme fieldConfigScheme);

    /**
     * Gets {@link FieldConfigScheme} for given {@code project}.
     *
     * @param project Project
     * @return {@link FieldConfigScheme} associated with given {@code project}
     */
    FieldConfigScheme getScheme(@Nullable Project project);

    /**
     * Gets {@link FieldConfigScheme} by {@code id}.
     *
     * @param id scheme id
     * @return {@link FieldConfigScheme}
     */
    @Nullable
    FieldConfigScheme getScheme(@Nonnull Long id);

    /**
     * Retrieves all priority schemes and sorts them.
     *
     * Schemes with the default scheme ID are first.
     * Schemes with null names are second, the rest are sorted on name.
     *
     * @return all schemes sorted on name with default first
     */
    @Nonnull
    List<FieldConfigScheme> getAllSchemes();

    /**
     * Checks if a passed {@code fieldConfigScheme} is the default priority scheme.
     *
     * @param fieldConfigScheme Field config scheme in question
     * @return true if {@code fieldConfigScheme} is default one, false otherwise.
     */
    boolean isDefaultScheme(@Nonnull FieldConfigScheme fieldConfigScheme);

    /**
     * Retrieves the default global priority scheme which contains all priorities.
     *
     * @return default {@link FieldConfigScheme}
     */
    @Nullable
    FieldConfigScheme getDefaultScheme();

    /**
     * Retrieves the ID of the default {@link Priority} for {@link FieldConfig} relevant to a passed {@code issueContext} for a priority system field.
     * If no default priority is defined for {@link FieldConfig} then middle priority of all priorities assigned to that {@link FieldConfig} is returned.
     *
     * @param issueContext {@link IssueContext} to obtain default value for
     * @return Id of default {@link Priority} for {@code issueContext}
     */
    @Nullable
    String getDefaultOption(@Nonnull IssueContext issueContext);

    /**
     * Returns the ID of the default {@link Priority} for a given {@code fieldConfig}.
     *
     * @param fieldConfig the field configuration scheme of interest
     * @return Id of default {@link Priority} for this {@code fieldConfig}
     */
    @Nullable
    String getDefaultOption(@Nullable FieldConfig fieldConfig);

    /**
     * Sets default priority ID for given {@link FieldConfigScheme}.
     *
     * @param fieldConfig {@link FieldConfig} for which set the default option
     * @param optionId    Priority id. Pass {@code null} to remove default priority from {@code fieldConfig}.
     */
    void setDefaultOption(@Nonnull FieldConfig fieldConfig, @Nullable String optionId);

    /**
     * Retrieves a list of priority IDs configured for a passed {@code issueContext}.
     * If {@code null} is passed for {@code issueContext} then priority ids for default scheme are returned.
     *
     * @param issueContext Issue context eg. {@link com.atlassian.jira.issue.context.ProjectContext}
     * @return list of priorities configured for passed {@code issueContext}.
     */
    List<String> getOptions(@Nullable IssueContext issueContext);

    /**
     * Retrieves a list of priority IDs configured for a passed {@code fieldConfig}.
     * If {@code null} is passed as {@code fieldConfig} then empty list is returned.
     *
     * @param fieldConfig {@link FieldConfig} to obtain list of priority id from.
     * @return List of priority ids for {@code fieldConfig}
     */
    @Nonnull
    List<String> getOptions(@Nullable FieldConfig fieldConfig);

    /**
     * Sets priority IDs for {@code fieldConfig}.
     *
     * @param fieldConfig {@link FieldConfig} to set list of priority id for.
     * @param optionIds   List of priority ids.
     */
    void setOptions(@Nonnull FieldConfig fieldConfig, @Nullable List<String> optionIds);

    /**
     * Adds the {@code optionId} to the default priority scheme.
     *
     * @param optionId priority id
     */
    void addOptionToDefault(@Nonnull String optionId);

    /**
     * Removes a priority ID from all priority schemes that have the passed {@code optionId} as part of its options.
     *
     * @param optionId Priority id to remove from all priority schemes.
     */
    void removeOptionFromAllSchemes(@Nonnull String optionId);

    /**
     * Retrieves all schemes that have {@code optionId} as part of its options.
     *
     * @param optionId Priority id being queried
     * @return Collection of {@link FieldConfigScheme} objects
     */
    @Nonnull
    Collection<FieldConfigScheme> getAllRelatedSchemes(@Nonnull String optionId);

    /**
     * Converts {@code priorityId} to {@link Priority} value.
     *
     * @param priorityId Id of {@link Priority}
     * @return {@link Priority} object
     */
    @Nullable
    Priority getPriority(@Nonnull String priorityId);

    /**
     * Converts a collection of priority IDs to a collection of {@link Priority}. Priorities not found by id are skipped.
     *
     * @param priorityIds Collection of ids for {@link Priority}
     * @return Collection of {@link Priority} objects
     */
    @Nonnull
    Collection<Priority> getPrioritiesFromIds(@Nonnull Collection<String> priorityIds);

    /**
     * Associates a project with a priority scheme.
     *
     * To assign project to "default priority scheme" please unassign project from currently used priority scheme {@link PrioritySchemeManager#unassignProject(FieldConfigScheme, Project)}.
     * If this method is used on "default priority scheme" it will remove DB association that marks scheme as default thus it will break "default priority scheme".
     *
     * When assigning multiple projects please use {@link PrioritySchemeManager#assignProjects(FieldConfigScheme, Set)} instead of calling this method in a loop to achieve better performance.
     *
     * @param priorityFieldConfig the scheme
     * @param project             the project
     */
    void assignProject(@Nonnull FieldConfigScheme priorityFieldConfig, @Nonnull Project project);

    /**
     * Associates multiple projects with a priority scheme.
     *
     * To assign project to "default priority scheme" please unassign project from currently used priority scheme {@link PrioritySchemeManager#unassignProject(FieldConfigScheme, Project)}.
     * If this method is used on "default priority scheme" it will remove DB association that marks scheme as default thus it will break "default priority scheme".
     *
     * @param priorityFieldConfig the scheme
     * @param projects            the projects
     * @since 7.6.13, 7.13.3, 8.0.3, 8.1.0 (be careful when checking if this method can be used, as it was added in a bugfix release)
     */
    void assignProjects(@Nonnull FieldConfigScheme priorityFieldConfig, @Nonnull Set<Project> projects);

    /**
     * Retrieves all projects associated with a scheme.
     *
     * @param fieldConfigScheme the scheme
     * @return List of {@link Project} with the given scheme (possibly empty)
     */
    @Nonnull
    Set<Project> getProjectsWithScheme(@Nonnull FieldConfigScheme fieldConfigScheme);

    /**
     * Removes a project from a priority scheme.
     *
     * It can also be used to assign project with "default priority scheme" - simply unassign project from currently used scheme and the project will start using default priority scheme.
     *
     * When unassigning multiple projects please use {@link PrioritySchemeManager#unassignProjects(FieldConfigScheme, Set)} instead of calling this method in a loop to achieve better performance.
     *
     * @param scheme  the scheme
     * @param project the project
     */
    void unassignProject(@Nonnull FieldConfigScheme scheme, @Nonnull Project project);

    /**
     * Removes multiple projects from a priority scheme.
     * 
     * It can also be used to assign projects with "default priority scheme" - simply unassign projects from currently used scheme and the projects will start using default priority scheme.
     *
     * @param scheme   the scheme
     * @param projects the projects
     * @since 7.6.13, 7.13.3, 8.0.3, 8.1.0 (be careful when checking if this method can be used, as it was added in a bugfix release)
     */
    void unassignProjects(@Nonnull FieldConfigScheme scheme, @Nonnull Set<Project> projects);
}
