package com.atlassian.jira.issue.customfields.vdi;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.issue.Issue;
import org.apache.lucene.document.Document;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Provides a collection of nonnull custom fields of a given issue. The {@link #getIdentity()} will be used
 * for deduplication of a provider that have multiple instances (e.g. Spring Proxies around the core instance).
 * See also {@link com.atlassian.jira.component.IdentifiableComponent}.
 *
 * @since 8.12
 */
@ExperimentalApi
public interface NonNullCustomFieldProvider {
    /**
     * @param issue the id of the issue
     * @return a map, where the keys are a collection of nonnull value custom field associated with the given issue and known to this provider
     * and the nullable values are prefetched data objects passed on to the indexer of this custom field. See {@link com.atlassian.jira.issue.index.indexers.FieldIndexer#addIndex(Document, Issue, CustomFieldPrefetchedData)}
     * @deprecated since 8.12, use <code>getCustomFieldInfo(List<Issue> issues)</code> instead
     */
    @Deprecated
    default Map<String, CustomFieldPrefetchedData> getCustomFieldInfo(Issue issue) {
        throw new UnsupportedOperationException();
    }

    /**
     * @param issues list of the issues
     * @return a map, where each issue is linked to a map, where the keys are a collection of nonnull value custom field associated with the given issue and known to this provider
     * and the nullable values are prefetched data objects passed on to the indexer of this custom field. See {@link com.atlassian.jira.issue.index.indexers.FieldIndexer#addIndex(Document, Issue, CustomFieldPrefetchedData)}
     */
    default Map<Long, Map<String, CustomFieldPrefetchedData>> getCustomFieldInfo(final List<Issue> issues) {
        return issues.stream()
                .collect(Collectors.toMap(Issue::getId, this::getCustomFieldInfo));
    }
    /**
     * @return the identity of this provider. The identity will be used for equality check to ensure that any proxy of
     * a provider will equal to it.
     */
    Object getIdentity();
}
