package com.atlassian.jira.issue.customfields;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.vdi.CustomFieldPrefetchedData;
import com.atlassian.jira.issue.index.indexers.FieldIndexer;
import org.apache.lucene.document.Document;

import java.util.List;

/**
 * An interface that can be implemented by a custom field type. By implementing this interface a custom field
 * type lets JIRA 'know' that it can potentially be used as a 'Group Selector'. JIRA then uses this field in places where
 * a group selector can be shown. For example:
 * <ul>
 * <li>The {@link com.atlassian.jira.security.type.GroupCF} Security type uses a custom field of type 'Group Selector'
 * to grant permissions.</li>
 * <li>The {@link com.atlassian.jira.workflow.condition.InGroupCFCondition} uses a custom field of type 'Group
 * Selector' to determine if a user is allowed to execute a workflow transition.
 * </ul>
 * <p>
 * A Group Selector is any field that can return a group name or a group id (String) as its value.
 * <p>
 * If a {@link CustomFieldType} implements this interface it should return values as objects of one of the following
 * types:
 * <ul>
 * <li>{@link com.atlassian.crowd.embedded.api.Group}</li>
 * <li>{@link java.util.Collection} of {@link com.atlassian.crowd.embedded.api.Group}s</li>
 * <li>{@link String}</li>
 * <li>{@link java.util.Collection} of {@link String}s</li>
 * </ul>
 * </p>
 * <p>
 * For example, a Multi Select custom field is a Group Selector as the values it returns are a Collection of Strings. These
 * strings <b>must</b> represent group names. It is up to JIRA Administrator to ensure that the when the Multi Select
 * custom field's options are defined, that all of them map exactly to a group name.
 */
public interface GroupSelectorField {

    /**
     * This method should be implemented in your custom type to return a List of identifiers of your group.
     *
     * Usually the identifier is just the {@code groupName}. However, some custom fields, such as {@link SelectCFType}
     * assign ids to their values and index the ids. In such a case, the custom field should return all the ids
     * under which the {@code groupName} was indexed. These should match the values that were used in the implementation of
     * {@link FieldIndexer#addIndex(Document, Issue, CustomFieldPrefetchedData)}
     *
     * @param groupName The name of the group
     * @return          The group identifier that will be used to search for this group in the index
     */
    List<String> getIdentifiersForGroup(String groupName);
}
