package com.atlassian.jira.issue.archiving;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.bc.ServiceResultImpl;
import com.atlassian.jira.event.issue.IssuesArchivedEvent;
import com.atlassian.jira.event.issue.IssuesRestoredEvent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;

import javax.annotation.Nonnull;

/**
 *  This service can be used for archiving and restoring individual issues.
 *
 *  @since 8.1
 */
@ExperimentalApi
public interface ArchivedIssueService {
    /**
     * Mark the issue as archived and perform operation of archiving including all dependencies.
     * This method will also trigger {@link IssuesArchivedEvent}.
     *
     * @param validation a request previously validated by
     *                {@link #validateArchiveIssue(ApplicationUser, String, boolean)}.
     * @return the result of the operation.
     */
    ArchivingResult archiveIssue(ValidationResult validation);

    /**
     * Mark the issue as not archived and perform operation of restoring including all dependencies.
     * This method will also trigger {@link IssuesRestoredEvent}.
     *
     * @param validation a request previously validated by
     *                {@link #validateRestoreIssue(ApplicationUser, String, boolean)}.
     * @return the result of the operation.
     */
    ArchivingResult restoreIssue(ValidationResult validation);

    /**
     * Check if given issue can be archived by given user.
     *
     * @param user
     * @param issueKey
     * @param notifyUsers
     * @return the validation result.
     */
    ValidationResult validateArchiveIssue(ApplicationUser user, String issueKey, boolean notifyUsers);

    /**
     * Check if given issue can be restored by given user.
     *
     * @param user
     * @param issueKey
     * @param notifyUsers
     * @return the validation result.
     */
    ValidationResult validateRestoreIssue(ApplicationUser user, String issueKey, boolean notifyUsers);

    /**
     * A simple object that holds the information about an issue archive/restore operation. This object should not be constructed
     * directly.
     */
    class ArchivingResult extends ValidationResult {
        public ArchivingResult(ErrorCollection errorCollection, @Nonnull Issue issue) {
            super(errorCollection, issue);
        }
    }

    /**
     * A simple object that holds the information about validating an archive/restore issue operation. This object should not be
     * constructed directly, you should invoke the {@link #validateArchiveIssue(ApplicationUser, String, boolean)} method
     * to obtain this.
     */
    class ValidationResult extends ServiceResultImpl {
        private final String issueKey;
        private final Issue issue;
        private final ApplicationUser user;
        private final boolean notifyUsers;

        public ValidationResult(ErrorCollection errorCollection, String issueKey) {
            super(errorCollection);
            this.issueKey = issueKey;
            this.issue = null;
            this.user = null;
            this.notifyUsers = true;
        }

        public ValidationResult(ErrorCollection errorCollection, @Nonnull Issue issue) {
            this(errorCollection, issue, null, true);
        }

        /**
         * @deprecated please use {@link #ValidationResult(ErrorCollection, Issue, ApplicationUser, boolean)}
         */
        @Deprecated
        public ValidationResult(ErrorCollection errorCollection, @Nonnull Issue issue, ApplicationUser user) {
            this(errorCollection, issue, user, true);
        }

        public ValidationResult(ErrorCollection errorCollection, @Nonnull Issue issue, ApplicationUser user, boolean notifyUsers) {
            super(errorCollection);
            this.issueKey = issue.getKey();
            this.issue = issue;
            this.user = user;
            this.notifyUsers = notifyUsers;
        }

        public String getIssueKey() {
            return issueKey;
        }

        public Issue getIssue() {
            return issue;
        }

        public ApplicationUser getUser() {
            return user;
        }

        public boolean isNotifyUsers() {
            return notifyUsers;
        }
    }
}
