package com.atlassian.jira.index.summary;

import com.atlassian.annotations.ExperimentalApi;
import java.time.Instant;

/**
 * Summary for the issue index, which contains data needed to diagnose the issue indexing status.
 *
 * If {@link #isIndexReadable()} returns {@code false} other fields are not valid.
 *
 * @since v7.2.10
 */
@ExperimentalApi
public class IssueIndexSummary {

    private final boolean indexReadable;

    private final long countInDatabase;

    private final long countInIndex;

    private final long countInArchive;

    private final Instant lastUpdatedInDatabase;

    private final Instant lastUpdatedInIndex;

    /**
     * Returns an instance of {@link IssueIndexSummary} that represents a not readable issue index.
     *
     * @return summary of not readable issue index
     */
    public static IssueIndexSummary notReadable() {
        return new IssueIndexSummary(false, 0, 0, 0, null, null);
    }

    /**
     * Constructor with all object fields.
     *
     * @param indexReadable is issue index readable
     * @param countInDatabase number of issues in database
     * @param countInIndex number of issues indexed
     * @param countInArchive number of archived issues
     * @param lastUpdatedInDatabase last issue update time in database
     * @param lastUpdatedInIndex last issue update time in index
     */
    public IssueIndexSummary(final boolean indexReadable,
                             final long countInDatabase,
                             final long countInIndex,
                             final long countInArchive,
                             final Instant lastUpdatedInDatabase,
                             final Instant lastUpdatedInIndex) {
        this.indexReadable = indexReadable;
        this.countInDatabase = countInDatabase;
        this.countInIndex = countInIndex;
        this.countInArchive = countInArchive;
        this.lastUpdatedInDatabase = lastUpdatedInDatabase;
        this.lastUpdatedInIndex = lastUpdatedInIndex;
    }

    /**
     * Was issue index readable. When this returns {@code false} other fields of this object are not valid.
     *
     * @return {@code true} if issue index is readable, {@code false} otherwise
     */
    public boolean isIndexReadable() {
        return indexReadable;
    }

    /**
     * Returns the number of issues found in the database.
     *
     * @return number of issues in database
     */
    public long getCountInDatabase() {
        return countInDatabase;
    }

    /**
     * Returns the number of issues found in the index.
     *
     * @return number of issues in index
     */
    public long getCountInIndex() {
        return countInIndex;
    }

    /**
     * Returns the number of archived issues.
     *
     * @return number of archived issues
     */
    public long getCountInArchive() {
        return countInArchive;
    }

    /**
     * Returns the update time of the last updated issue found in the database.
     *
     * @return last issue update time in database
     */
    public Instant getLastUpdatedInDatabase() {
        return lastUpdatedInDatabase;
    }

    /**
     * Returns the update time of the last updated issue found in the index.
     *
     * @return last issue update time in index
     */
    public Instant getLastUpdatedInIndex() {
        return lastUpdatedInIndex;
    }
}
