package com.atlassian.jira.event.user.anonymize;

import com.atlassian.analytics.api.annotations.EventName;

import javax.annotation.Nonnull;
import java.util.Objects;

/**
 * This event is sent when a user name is changed.
 *
 * @since v8.4
 */
@EventName("jira.user.name.changed")
public class UserNameChangedEvent {
    private final String oldUserName;
    private final String newUserName;

    /**
     * @param oldUserName old user name (before the change)
     * @param newUserName new user name (after the change)
     */
    public UserNameChangedEvent(@Nonnull String oldUserName, @Nonnull String newUserName) {
        this.oldUserName = Objects.requireNonNull(oldUserName);
        this.newUserName = Objects.requireNonNull(newUserName);
    }

    /**
     * Returns old user name (before the change).
     *
     * @return old user name (before the change).
     */
    @Nonnull
    public String getOldUserName() {
        return oldUserName;
    }

    /**
     * Returns new user name (after the change).
     *
     * @return new user name (after the change).
     */
    @Nonnull
    public String getNewUserName() {
        return newUserName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UserNameChangedEvent that = (UserNameChangedEvent) o;
        return Objects.equals(oldUserName, that.oldUserName) &&
                Objects.equals(newUserName, that.newUserName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(oldUserName, newUserName);
    }
}
